package ru.yandex.direct.intapi.entity.uac.controller

import io.swagger.annotations.Api
import io.swagger.annotations.ApiOperation
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.http.HttpStatus
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RestController
import org.springframework.http.MediaType
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.PostMapping
import org.springframework.web.bind.annotation.RequestBody
import ru.yandex.direct.core.entity.uac.service.appinfo.ParseAppStoreContentService
import ru.yandex.direct.core.entity.uac.service.appinfo.ParseAppStoreUrlService
import ru.yandex.direct.core.entity.uac.validation.invalidAppUrl
import ru.yandex.direct.core.service.appinfo.UacAppInfoParsingService
import ru.yandex.direct.i18n.Language
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService
import ru.yandex.direct.tvm.AllowServices
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.web.entity.uac.model.AppInfoWithContent

import ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER
import ru.yandex.direct.tvm.TvmService.RSYA_PARTNER_INTERFACE_PROD
import ru.yandex.direct.tvm.TvmService.RSYA_PARTNER_INTERFACE_TEST
import ru.yandex.direct.web.entity.uac.model.ParseAppInfoRequestWithLanguage


@RestController
@Api(value = "API для получения информации о uac")
@RequestMapping("/uac", produces = [MediaType.APPLICATION_JSON_VALUE])
open class IntapiAppInfoParsingController(
    private val  parseAppStoreUrlService: ParseAppStoreUrlService,
    private val validationResultConversionService: ValidationResultConversionService,
    private val appInfoParsingService: UacAppInfoParsingService,
) {
    companion object {
        private val logger: Logger = LoggerFactory.getLogger(IntapiAppInfoParsingController::class.java)
        private val DEFAULT_LANGUAGE = Language.RU
    }

    @ApiOperation(
        value = "appInfoParsing",
        httpMethod = "POST",
        nickname = "appInfoParsing",
    )
    @PostMapping(
        value = ["app_info/parsing"],
        consumes = [MediaType.APPLICATION_JSON_UTF8_VALUE],
        produces = [MediaType.APPLICATION_JSON_UTF8_VALUE],
    )
    @AllowServices(production = [RSYA_PARTNER_INTERFACE_PROD], testing = [RSYA_PARTNER_INTERFACE_TEST, DIRECT_DEVELOPER])
    open fun parseFromStoreOrTrackingUrl(
        @RequestBody request: ParseAppInfoRequestWithLanguage
    ): ResponseEntity<Any> {
        val appStoreUrlValidation = parseAppStoreUrlService.getAppStoreUrl(request.url)
        if (appStoreUrlValidation.hasErrors()) {
            return ResponseEntity(
                validationResultConversionService.buildValidationResponse(
                    appStoreUrlValidation
                ),
                HttpStatus.BAD_REQUEST
            )
        }
        val url = appStoreUrlValidation.value

        try {
            val (appInfo, contents) = appInfoParsingService.getFromCacheOrParse(
                url,
                request.cacheTtl ?: -1
            )
            val regionWithName = appInfoParsingService.getDefaultTargetRegionWithName(
                region = url.parsedUrl.storeCountry,
                lang = (request.language ?: DEFAULT_LANGUAGE).toString()
            )

            return ResponseEntity.ok(
                AppInfoWithContent(
                    appInfo = appInfo,
                    contents = contents,
                    isTrackingUrl = url.fromTrackingUrl,
                    defaultTargetRegion = regionWithName?.first,
                    defaultTargetRegionName = regionWithName?.second,
                )
            )
        } catch (e: ParseAppStoreContentService.ParseAppStoreContentException) {
            logger.error("Failed to parse store", e)
            return ResponseEntity(validationResultConversionService.buildValidationResponse(
                ValidationResult.failed(null, invalidAppUrl())
            ),
                HttpStatus.BAD_REQUEST)
        }
    }
}
