package ru.yandex.direct.intapi.entity.user.service;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.validator.routines.EmailValidator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.eventlog.service.EventLogService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.notification.repository.SmsQueueRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.intapi.entity.user.model.SendMoneyOutWarningRequestItem;
import ru.yandex.direct.intapi.entity.user.model.UserInfoResponse;
import ru.yandex.direct.intapi.entity.user.model.UserWarningsMailNotificationType;
import ru.yandex.direct.sender.YandexSenderClient;
import ru.yandex.direct.sender.YandexSenderException;
import ru.yandex.direct.sender.YandexSenderTemplateParams;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;

import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Service
@ParametersAreNonnullByDefault
public class IntapiUserService {

    private static final EmailValidator EMAIL_VALIDATOR = EmailValidator.getInstance();
    private static final Logger logger = LoggerFactory.getLogger(IntapiUserService.class);

    private final YandexSenderClient senderClient;
    private final UserWarningsMailTemplateResolver templateResolver;
    private final UserService userService;
    private final SmsQueueRepository smsQueueRepository;
    private final ShardHelper shardHelper;
    private final FeatureService featureService;
    private final CampaignRepository campaignRepository;
    private final EventLogService eventLogService;

    private final SmsHelper smsHelper;
    private final CampaignNotificationFilter campaignNotificationFilter;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public IntapiUserService(
            UserService userService,
            YandexSenderClient senderClient,
            UserWarningsMailTemplateResolver templateResolver,
            SmsQueueRepository smsQueueRepository,
            ShardHelper shardHelper,
            TranslationService translationService,
            CampaignRepository campaignRepository,
            ClientRepository clientRepository,
            EventLogService eventLogService,
            AdGroupRepository adGroupRepository,
            FeatureService featureService) {
        this.userService = userService;
        this.senderClient = senderClient;
        this.templateResolver = templateResolver;
        this.smsQueueRepository = smsQueueRepository;
        this.shardHelper = shardHelper;
        this.featureService = featureService;
        this.campaignRepository = campaignRepository;
        this.eventLogService = eventLogService;

        smsHelper = new SmsHelper(translationService, shardHelper, clientRepository);
        campaignNotificationFilter = new CampaignNotificationFilter(shardHelper, adGroupRepository);
    }

    public WebResponse getInfo(Long userId) {
        try {
            boolean isOfferAccepted = userService.getIsOfferAccepted(userId);
            return new UserInfoResponse(isOfferAccepted);
        } catch (IllegalArgumentException ignored) {
            return new WebErrorResponse(1, "User not found");
        }
    }

    public Map<Long, Boolean> sendMoneyOutWarnings(List<SendMoneyOutWarningRequestItem> warningItems) {
        final Map<Long, Campaign> campaignsByCid = shardHelper.groupByShard(warningItems,
                ShardKey.CID, SendMoneyOutWarningRequestItem::getCid)
                .stream()
                .map(e -> campaignRepository.getCampaigns(e.getKey(),
                        listToSet(e.getValue(), SendMoneyOutWarningRequestItem::getCid)))
                .flatMap(Collection::stream)
                .toMap(Campaign::getId, Function.identity());

        final Set<Long> campaignsIdToSendNotifications = campaignNotificationFilter
                .shouldSendNotification(campaignsByCid.values());

        campaignsByCid.keySet().retainAll(campaignsIdToSendNotifications);

        final Set<Long> userIds = warningItems
                .stream()
                .filter(paramsItem -> campaignsIdToSendNotifications.contains(paramsItem.getCid()))
                .map(SendMoneyOutWarningRequestItem::getUid).collect(Collectors.toSet());

        final Map<Long, User> userByUids = userService.massGetUser(userIds)
                .stream()
                .collect(Collectors.toMap(User::getId, Function.identity()));

        final Set<Long> touchCampaignsIds = smsHelper.getTouchCampaignsIds(campaignsByCid.values());

        return StreamEx.of(warningItems)
                .mapToEntry(SendMoneyOutWarningRequestItem::getCid,
                        paramsItem -> !campaignsIdToSendNotifications.contains(paramsItem.getCid()) ||
                                trySendNotifications(
                                        userByUids.get(paramsItem.getUid()),
                                        campaignsByCid.get(paramsItem.getCid()),
                                        paramsItem,
                                        touchCampaignsIds.contains(paramsItem.getCid())))
                .toMap();
    }

    private boolean trySendNotifications(User user,
                                         Campaign campaign,
                                         SendMoneyOutWarningRequestItem paramsItem,
                                         boolean isTouch) {
        try {
            sendMailIfNeeded(user, paramsItem.getCustomEmail(),
                    UserWarningsMailNotificationType.MONEY_OUT_WARNING, paramsItem.getTemplateData());
            trySendSms(user, campaign, isTouch);
            eventLogService.addMoneyOutEventLog(campaign.getId(), campaign.getType(), campaign.getClientId());
            return true;
        } catch (YandexSenderException e) {
            return false;
        }
    }

    private void sendMailIfNeeded(User user,
            @Nullable String customEmail,
            UserWarningsMailNotificationType notificationType,
            @Nullable Map<String, String> customTemplateParams) {

        String email = nvl(customEmail, user.getEmail());
        if (!EMAIL_VALIDATOR.isValid(email)) {
            logger.error("User: {} - invalid email: {}", user.getLogin(), email);
            return;
        }
        String campSlug = templateResolver.resolveTemplateId(user.getLang(), notificationType);
        if (campSlug == null || campSlug.isEmpty()) {
            logger.error("Couldn't resolve template slug for notification type {} user {} lang {}",
                    notificationType.name(), user.getLogin(), user.getLang());
            return;
        }
        //В args обязательно должен быть ClientID
        if (customTemplateParams == null) {
            customTemplateParams = new HashMap<>();
        }
        customTemplateParams.put("ClientID", user.getClientId().toString());

        YandexSenderTemplateParams templateParams = new YandexSenderTemplateParams.Builder()
                .withCampaignSlug(campSlug)
                .withToEmail(email)
                .withAsync(Boolean.TRUE)
                .withArgs(customTemplateParams)
                .build();
        try {
            if (senderClient.sendTemplate(templateParams, YandexSenderClient::isInvalidToEmail)) {
                logger.info("mail sent - user {}, notification {}, slug {}",
                        user.getLogin(), notificationType.name(), campSlug);
            } else {
                logger.error("User: {} - unreachable email: {}", user.getLogin(), email);
            }
        } catch (YandexSenderException e) {
            logger.error("Sender error - user {}, email {}, notification {}, slug {}, message {}",
                    user.getLogin(), email, notificationType.name(), campSlug, e.getMessage());
            throw e;
        }
    }

    private void trySendSms(User user, Campaign campaign, boolean isTouch) {
        int shard = shardHelper.getShardByUserId(user.getId());
        boolean useDynamicThreshold = featureService.isEnabledForClientId(
                ClientId.fromLong(campaign.getClientId()),
                FeatureName.USE_DYNAMIC_THRESHOLD_FOR_SEND_ORDER_WARNINGS);

        SmsHelper.UserWarningSms message = smsHelper.createSmsMessage(user, campaign, isTouch, useDynamicThreshold);
        smsQueueRepository.addToSmsQueue(shard, user.getUid(), campaign.getId(), message.getSmsText(),
                message.getSmsFlag(), campaign.getSmsTime());
    }
}
