package ru.yandex.direct.intapi.entity.user.service;

import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignOpts;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.SmsFlag;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.notification.LocaleResolver;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;

import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class SmsHelper {

    private final TranslationService translationService;
    private final ShardHelper shardHelper;
    private final ClientRepository clientRepository;

    public SmsHelper(TranslationService translationService,
                     ShardHelper shardHelper,
                     ClientRepository clientRepository) {
        this.translationService = translationService;
        this.shardHelper = shardHelper;
        this.clientRepository = clientRepository;
    }

    public static final class UserWarningSms {
        private final String smsText;
        private final SmsFlag smsFlag;

        public UserWarningSms(String smsText, SmsFlag smsFlag) {
            this.smsText = smsText;
            this.smsFlag = smsFlag;
        }

        public String getSmsText() {
            return smsText;
        }

        public SmsFlag getSmsFlag() {
            return smsFlag;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            UserWarningSms that = (UserWarningSms) o;
            return Objects.equals(smsText, that.smsText) && smsFlag == that.smsFlag;
        }

        @Override
        public int hashCode() {
            return Objects.hash(smsText, smsFlag);
        }

        @Override
        public String toString() {
            return "UserWarningSms{" +
                    "smsText='" + smsText + '\'' +
                    ", smsFlag=" + smsFlag +
                    '}';
        }
    }

    public Set<Long> getTouchCampaignsIds(Collection<Campaign> campaigns) {
        List<Long> clientsIdsToCheckForTouch = campaigns.stream()
                .filter(campaign -> !campaign.getOpts().contains(CampaignOpts.IS_TOUCH) &&
                        campaign.getType() == CampaignType.WALLET)
                .map(Campaign::getClientId)
                .filter(Objects::nonNull)
                .distinct()
                .collect(Collectors.toList());

        Set<Long> touchWalletClientsIds = shardHelper.groupByShard(clientsIdsToCheckForTouch, ShardKey.CLIENT_ID)
                .stream()
                .map(e -> clientRepository.get(e.getKey(), mapList(e.getValue(), ClientId::fromLong)))
                .flatMap(Collection::stream)
                .filter(client -> Boolean.TRUE.equals(client.getIsTouch()))
                .map(Client::getId)
                .toSet();

        return filterAndMapToSet(campaigns,
                campaign -> isTouchCampaign(campaign, touchWalletClientsIds),
                Campaign::getId);
    }


    private static boolean isTouchCampaign(Campaign campaign, final Set<Long> touchWalletClientsIds) {
        if (campaign == null) {
            return false;
        }
        if (campaign.getOpts().contains(CampaignOpts.IS_TOUCH)) {
            return true;
        }
        if (campaign.getType() != CampaignType.WALLET) {
            return false;
        }
        Long clientId = campaign.getClientId();
        if (clientId == null) {
            return false;
        }
        return touchWalletClientsIds.contains(clientId);
    }

    public UserWarningSms createSmsMessage(User user, Campaign campaign, boolean isTouch, boolean useDynamicThreshold) {
        final Locale locale = LocaleResolver.getLocaleByLanguageWithFallback(user.getLang());
        ActiveOrdersMoneyOutSmsTranslations instance = ActiveOrdersMoneyOutSmsTranslations.INSTANCE;

        if (useDynamicThreshold) {
            String clientLogin = user.getLogin();
            if (isTouch) {
                return new UserWarningSms(
                        translationService.translate(
                                instance.activeOrdersMoneyOutCampaignStoppedTouch(clientLogin), locale),
                        SmsFlag.ACTIVE_ORDERS_MONEY_OUT_CAMPAIGN_STOPPED_TOUCH_SMS);
            } else {
                return new UserWarningSms(
                        translationService.translate(instance.activeOrdersMoneyOutCampaignStopped(clientLogin), locale),
                        SmsFlag.ACTIVE_ORDERS_MONEY_OUT_CAMPAIGN_STOPPED_SMS);
            }
        } else {
            if (isTouch) {
                return new UserWarningSms(
                        translationService.translate(instance.activeOrdersMoneyOutTouch(), locale),
                        SmsFlag.ACTIVE_ORDERS_MONEY_OUT_TOUCH_SMS);
            } else if (campaign.getType() == CampaignType.WALLET) {
                return new UserWarningSms(
                        translationService.translate(instance.activeOrdersMoneyOutWallet(), locale),
                        SmsFlag.ACTIVE_ORDERS_MONEY_OUT_SMS);
            } else {
                return new UserWarningSms(
                        translationService.translate(instance.activeOrdersMoneyOut(campaign.getId()), locale),
                        SmsFlag.ACTIVE_ORDERS_MONEY_OUT_SMS);
            }
        }
    }
}
