package ru.yandex.direct.intapi.entity.userphone;

import java.util.List;
import java.util.Map;
import java.util.Objects;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.intapi.entity.userphone.model.CheckVerifiedBulkRequest;
import ru.yandex.direct.intapi.entity.userphone.model.CheckVerifiedBulkResponse;
import ru.yandex.direct.intapi.entity.userphone.model.CheckVerifiedRequest;
import ru.yandex.direct.intapi.entity.userphone.model.CheckVerifiedResponse;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxPhone;

import static java.util.Collections.emptyList;
import static org.springframework.http.MediaType.APPLICATION_JSON_UTF8_VALUE;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_TEST;

@RestController
@RequestMapping(
        path = "/user_phones",
        consumes = APPLICATION_JSON_UTF8_VALUE,
        produces = APPLICATION_JSON_UTF8_VALUE)
@AllowServices(production = {DIRECT_WEB_PROD}, testing = {DIRECT_WEB_TEST, DIRECT_DEVELOPER})
public class UserPhoneController {

    private static final Logger logger = LoggerFactory.getLogger(UserPhoneController.class);

    private final UserService userService;

    @Autowired
    public UserPhoneController(UserService userService) {
        this.userService = userService;
    }

    @RequestMapping(
            path = "/check_verified",
            method = RequestMethod.POST
    )
    public WebResponse checkVerified(@RequestBody CheckVerifiedRequest request) {
        Long uid = request.getUid();
        Long phoneId = request.getPhoneId();

        List<BlackboxPhone> allUserPhones = userService.getAllBlackboxPhones(uid);

        return convertToResponseItem(uid, phoneId, allUserPhones);
    }

    @RequestMapping(
            path = "/check_verified_bulk",
            method = RequestMethod.POST
    )
    public WebResponse checkVerifiedBulk(@RequestBody CheckVerifiedBulkRequest request) {
        Map<Long, Long> phoneIdByUid = StreamEx.of(request.getCheckVerifiedRequestItems())
                .toMap(CheckVerifiedRequest::getUid, CheckVerifiedRequest::getPhoneId);

        Map<Long, List<BlackboxPhone>> allPhonesByUid = userService.getAllBlackboxPhonesByUid(phoneIdByUid.keySet());

        List<CheckVerifiedResponse> responseItems = EntryStream.of(phoneIdByUid)
                .mapKeyValue((uid, phoneId) -> {
                    List<BlackboxPhone> allUserPhones = allPhonesByUid.getOrDefault(uid, emptyList());

                    return convertToResponseItem(uid, phoneId, allUserPhones);
                })
                .toList();

        return new CheckVerifiedBulkResponse()
                .withResponseItems(responseItems);
    }

    private CheckVerifiedResponse convertToResponseItem(Long uid, Long phoneId, List<BlackboxPhone> allUserPhones) {
        List<BlackboxPhone> requiredPhones = StreamEx.of(allUserPhones)
                .filter(blackboxPhone -> blackboxPhone.getIsConfirmedNumber().getOrElse(false))
                .filter(verifiedBlackboxPhone -> Objects.equals(verifiedBlackboxPhone.getPhoneId(), phoneId))
                .toList();

        if (requiredPhones.isEmpty()) {
            return new CheckVerifiedResponse()
                    .withUid(uid)
                    .withVerified(false)
                    .withPhone(null);
        } else {
            if (requiredPhones.size() > 1) {
                logger.error("More than one phone found for user {}", uid);
            }

            BlackboxPhone requiredPhone = requiredPhones.get(0);

            return new CheckVerifiedResponse()
                    .withUid(uid)
                    .withVerified(true)
                    .withPhone(requiredPhone.getMaskedE164Number().get());
        }
    }
}
