package ru.yandex.direct.intapi.entity.vcards.service;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.vcard.model.InstantMessenger;
import ru.yandex.direct.core.entity.vcard.model.Phone;
import ru.yandex.direct.core.entity.vcard.model.PointOnMap;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.core.entity.vcard.service.AddVcardOperation;
import ru.yandex.direct.core.entity.vcard.service.ManageVcardService;
import ru.yandex.direct.core.entity.vcard.service.VcardService;
import ru.yandex.direct.core.entity.vcard.service.validation.AddVcardValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidClientIdShard;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.intapi.entity.vcards.model.GetVcardsRequest;
import ru.yandex.direct.intapi.entity.vcards.model.GetVcardsResponse;
import ru.yandex.direct.intapi.entity.vcards.model.IntapiInstantMessenger;
import ru.yandex.direct.intapi.entity.vcards.model.IntapiPhone;
import ru.yandex.direct.intapi.entity.vcards.model.IntapiPointOnMap;
import ru.yandex.direct.intapi.entity.vcards.model.IntapiVcard;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResponse;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResult;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.vcard.VcardWorktimeUtils.toWebWorktimes;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class VcardsIntapiService {
    private final ShardHelper shardHelper;
    private final AddVcardValidationService addVcardValidationService;
    private final RbacService rbacService;
    private final ValidationResultConversionService validationResultConversionService;
    private final VcardService vcardService;
    private final ManageVcardService manageVcardService;


    @Autowired
    public VcardsIntapiService(ShardHelper shardHelper,
                               AddVcardValidationService addVcardValidationService,
                               RbacService rbacService,
                               ValidationResultConversionService validationResultConversionService,
                               VcardService vcardService,
                               ManageVcardService manageVcardService) {
        this.shardHelper = shardHelper;
        this.addVcardValidationService = addVcardValidationService;
        this.rbacService = rbacService;
        this.validationResultConversionService = validationResultConversionService;
        this.vcardService = vcardService;
        this.manageVcardService = manageVcardService;
    }


    public IntapiValidationResponse validateVcards(Long operatorUid, Long clientId, List<Vcard> vcardsList) {
        UidClientIdShard client = UidClientIdShard.of(
                rbacService.getChiefByClientId(ClientId.fromLong(clientId)),
                clientId,
                shardHelper.getShardByClientIdStrictly(ClientId.fromLong(clientId)));

        ValidationResult<List<Vcard>, Defect> validationResult =
                ListValidationBuilder.<Vcard, Defect>of(vcardsList)
                        .checkBy(vcards -> addVcardValidationService.validate(operatorUid,
                                false, vcards, client))
                        .getResult();

        if (validationResult != null && validationResult.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(validationResult);
        }
        return new IntapiValidationResponse(new IntapiValidationResult());
    }


    public List<Long> createVcards(Long operatorUid, Long clientId, List<Vcard> vcardsList) {
        // не проверяем права на кампанию
        // (необходимые проверки должны быть сделаны в перле, перед вызовом intapi)
        AddVcardOperation addOperation = vcardService
                .createFullAddVcardOperationAsPartOfComplexOperation(vcardsList, operatorUid,
                        ClientId.fromLong(clientId));
        MassResult<Long> result = addOperation.prepareAndApply();
        return result.getErrorCount() > 0 ? Collections.emptyList()
                : StreamEx.of(result.getResult()).map(Result::getResult).toList();
    }


    public GetVcardsResponse getVcards(Long operatorUid, Long clientId, GetVcardsRequest request) {
        // не проверяем права оператора на визитки
        // (необходимые проверки должны быть сделаны в перле, перед вызовом intapi)
        List<Vcard> vcards = manageVcardService.getVcards(null, ClientId.fromLong(clientId), request.getVcardIds());
        GetVcardsResponse response = new GetVcardsResponse()
                .withVcards(mapList(vcards, VcardsIntapiService::toIntapiVcard));

        if (request.getWithVcardsUsesCount()) {
            Map<Long, Long> vcardsUsesCount = manageVcardService.getVcardsUsesCount(
                    ClientId.fromLong(clientId), mapList(vcards, Vcard::getId));
            response.withVcardsUsesCount(vcardsUsesCount);
        }
        return response;
    }


    @Nullable
    private static IntapiVcard toIntapiVcard(@Nullable Vcard modelVcard) {
        return Optional.ofNullable(modelVcard)
                .map(vcard -> new IntapiVcard()
                        .withId(vcard.getId())
                        .withCampaignId(vcard.getCampaignId().toString())
                        .withUid(vcard.getUid().toString())
                        .withGeoId(vcard.getGeoId())
                        .withCountry(vcard.getCountry())
                        .withCity(vcard.getCity())
                        .withStreet(vcard.getStreet())
                        .withHouse(vcard.getHouse())
                        .withBuild(vcard.getBuild())
                        .withApart(vcard.getApart())
                        .withMetroId(vcard.getMetroId())
                        .withMetroName(vcard.getMetroName())
                        .withCountryGeoId(vcard.getCountryGeoId())
                        .withCompanyName(vcard.getCompanyName())
                        .withContactPerson((vcard.getContactPerson()))
                        .withEmail(vcard.getEmail())
                        .withWorkTime(vcard.getWorkTime())
                        .withWorkTimeList(toWebWorktimes(vcard.getWorkTime()))
                        .withPhone(toIntapiVcardPhone(vcard.getPhone()))
                        .withInstantMessenger(toIntapiVcardInstantMessenger(vcard.getInstantMessenger()))
                        .withExtraMessage(vcard.getExtraMessage())
                        .withOrgDetailsId(vcard.getOrgDetailsId())
                        .withOgrn(vcard.getOgrn())
                        .withManualPoint(toIntapiVcardPointOnMap(vcard.getManualPoint()))
                        .withAutoPoint(toIntapiVcardPointOnMap(vcard.getAutoPoint()))
                        .withPointType(vcard.getPointType())
                        .withPrecision(vcard.getPrecision()))
                .orElse(null);
    }

    @Nullable
    private static IntapiPhone toIntapiVcardPhone(@Nullable Phone phone) {
        return Optional.ofNullable(phone)
                .map(p -> new IntapiPhone()
                        .withCountryCode(p.getCountryCode())
                        .withCityCode(p.getCityCode())
                        .withPhoneNumber(p.getPhoneNumber())
                        .withExtension(p.getExtension()))
                .orElse(null);
    }

    @Nullable
    private static IntapiInstantMessenger toIntapiVcardInstantMessenger(@Nullable InstantMessenger messenger) {
        return Optional.ofNullable(messenger)
                .map(p -> new IntapiInstantMessenger()
                        .withType(p.getType())
                        .withLogin(p.getLogin()))
                .orElse(null);
    }

    @Nullable
    private static IntapiPointOnMap toIntapiVcardPointOnMap(@Nullable PointOnMap pointOnMap) {
        return Optional.ofNullable(pointOnMap)
                .map(p -> new IntapiPointOnMap()
                        .withId(p.getId())
                        .withX(p.getX())
                        .withY(p.getY())
                        .withX1(p.getX1())
                        .withY1(p.getY1())
                        .withX2(p.getX2())
                        .withY2(p.getY2()))
                .orElse(null);
    }
}
