package ru.yandex.direct.intapi.entity.video

import io.swagger.annotations.Api
import io.swagger.annotations.ApiOperation
import io.swagger.annotations.ApiResponse
import io.swagger.annotations.ApiResponses
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RequestMethod
import org.springframework.web.bind.annotation.RequestParam
import org.springframework.web.bind.annotation.RequestPart
import org.springframework.web.bind.annotation.ResponseBody
import org.springframework.web.multipart.MultipartFile
import ru.yandex.direct.canvas.client.CanvasClient
import ru.yandex.direct.canvas.client.model.exception.CanvasClientException
import ru.yandex.direct.canvas.client.model.video.VideoUploadResponse
import ru.yandex.direct.intapi.ErrorResponse
import ru.yandex.direct.tvm.AllowServices
import ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER
import ru.yandex.direct.tvm.TvmService.MILAB_PROD
import ru.yandex.direct.tvm.TvmService.MILAB_TEST
import ru.yandex.direct.web.core.model.WebResponse
import javax.ws.rs.core.MediaType

@Controller
@Api(value = "API для создания видео в Директе. Обертка для ручек канваса для внутренних сервисов Яндекса")
@RequestMapping(path = ["/video"])
@AllowServices(production = [MILAB_PROD], testing = [MILAB_TEST, DIRECT_DEVELOPER])
class VideoController @Autowired constructor(
    private val canvasClient: CanvasClient
) {

    companion object {
        private val logger = LoggerFactory.getLogger(VideoController::class.java)

        private val fileChecks: List<Pair<(MultipartFile) -> Boolean, CheckFileResult>> = listOf(
            { file: MultipartFile -> file.contentType.equals("video/mp4", true) } to CheckFileResult.BAD_FORMAT
        )

        data class CreationResponse(
            val creativeId: Long? = null,
            val message: String? = null,
        ) : WebResponse {
            override fun isSuccessful(): Boolean {
                return creativeId != null
            }
        }
    }

    @ApiOperation(
        value = "create-creative-from-video-file",
        nickname = "create-creative-from-video-file",
        httpMethod = "POST",
    )
    @ApiResponses(
        ApiResponse(code = 400, message = "Bad params", response = ErrorResponse::class),
        ApiResponse(code = 200, message = "Ok", response = CreationResponse::class),
    )
    @RequestMapping(
        path = ["/create-creative-from-video-file"],
        method = [RequestMethod.POST],
        consumes = [MediaType.MULTIPART_FORM_DATA],
        produces = [MediaType.APPLICATION_JSON],
    )
    @ResponseBody
    fun createCreativeFromVideoFile(
        @RequestParam clientId: Long,
        @RequestPart file: MultipartFile,
    ): CreationResponse {

        val checkFileResult = checkFile(file)
        if (!checkFileResult.validate()) {
            val error = processCheckFileError(file, checkFileResult)
            return CreationResponse(message = error.message)
        }

        val uploadResponse = uploadVideo(clientId, file)
            ?: return CreationResponse(message = "Failure to upload video from file, name = ${file.originalFilename}")

        val creativeId = createCreativeFromVideo(clientId, uploadResponse)
            ?: return CreationResponse(message = "Failure to create creative for videoId=${uploadResponse.id}")

        return CreationResponse(creativeId = creativeId)
    }

    private fun checkFile(file: MultipartFile): CheckFileResult {
        for (curCheck in fileChecks) {
            val result = curCheck.first(file)
            if (!result) {
                return curCheck.second
            }
        }

        return CheckFileResult.OK
    }

    private fun processCheckFileError(file: MultipartFile, result: CheckFileResult): ErrorResponse {
        return when (result) {
            CheckFileResult.OK -> throw IllegalArgumentException("CheckResult.OK is not an error")
            CheckFileResult.BAD_FORMAT -> {
                val message = result.getErrorMessage(file)
                logger.error(message)
                ErrorResponse(ErrorResponse.ErrorCode.BAD_PARAM, message)
            }
        }
    }

    private fun uploadVideo(clientId: Long, file: MultipartFile): VideoUploadResponse? {
        return try {
            val uploadResponse = canvasClient.createVideoFromFile(
                clientId, file.bytes, "3d_video", null, null
            )
            logger.info("Uploaded video id: {}", uploadResponse.id)
            uploadResponse
        } catch (e: CanvasClientException) {
            logger.error("Failure to upload video from file, name = ${file.originalFilename}", e)
            null
        }
    }

    private fun createCreativeFromVideo(clientId: Long, uploadResponse: VideoUploadResponse): Long? {
        return try {
            val createAdditionResponse =
                canvasClient.createDefaultAddition(clientId, uploadResponse.presetId, uploadResponse.id)
            logger.info("Created creative id: {}", createAdditionResponse.creativeId)
            createAdditionResponse.creativeId
        } catch (e: CanvasClientException) {
            logger.error("Failure to create creative for videoId=${uploadResponse.id}", e)
            null
        }
    }
}
