package ru.yandex.direct.intapi.logging;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.Arrays;

import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.RequestBody;

@Aspect
@Component
@SuppressWarnings("checkstyle:linelength")
public class RequestParametersLogAspect {

    private final RequestParametersLogWriter requestParametersLogWriter;

    @Autowired
    public RequestParametersLogAspect(RequestParametersLogWriter requestParametersLogWriter) {
        this.requestParametersLogWriter = requestParametersLogWriter;
    }

    @Pointcut("@within(org.springframework.stereotype.Controller) || @within(org.springframework.web.bind.annotation.RestController)")
    public void controllerClass() {
    }

    @Pointcut("@annotation(org.springframework.web.bind.annotation.RequestMapping)")
    public void requestMappingMethod() {
    }

    @Pointcut("within(ru.yandex.direct.intapi..*)")
    public void intapiPackages() {
    }

    /**
     * Срабатывает на методах контроллеров (класс должен быть помечен аннотацией
     * {@link org.springframework.stereotype.Controller}) или {@link org.springframework.web.bind.annotation.RestController}).
     * А также помеченных аннотацией {@link org.springframework.web.bind.annotation.RequestMapping},
     * если они расположены в ru.yandex.direct.intapi или ниже.
     * Если у метода есть аргумент с аннотацией {@link org.springframework.web.bind.annotation.RequestBody},
     * добавляет этот объект в запись лога текущего запроса.
     */
    @Before("controllerClass() && requestMappingMethod() && intapiPackages()")
    public void logRequestParams(JoinPoint joinPoint) {
        logOperatorUid(joinPoint);
        logClientId(joinPoint);
        logBodyRequest(joinPoint);
    }

    private void logOperatorUid(JoinPoint joinPoint) {
        int requestBodyIndex = getRequestParameterArgumentIndex(joinPoint, OperatorUid.class);
        if (requestBodyIndex == -1) {
            return;
        }

        Object requestBodyArgument = joinPoint.getArgs()[requestBodyIndex];
        requestParametersLogWriter.writeOperatorUidToLog(requestBodyArgument);
    }

    private void logClientId(JoinPoint joinPoint) {
        int requestBodyIndex = getRequestParameterArgumentIndex(joinPoint, ClientIdParam.class);
        if (requestBodyIndex == -1) {
            return;
        }

        Object requestBodyArgument = joinPoint.getArgs()[requestBodyIndex];
        requestParametersLogWriter.writeClientIdToLog(requestBodyArgument);
    }

    private void logBodyRequest(JoinPoint joinPoint) {
        int requestBodyIndex = getRequestParameterArgumentIndex(joinPoint, RequestBody.class);
        if (requestBodyIndex == -1) {
            return;
        }

        Object requestBodyArgument = joinPoint.getArgs()[requestBodyIndex];
        requestParametersLogWriter.writeBodyToLog(requestBodyArgument);
    }

    /**
     * Возвращает индекс аргумента с аннотацией {@code clazz}, если такой имеется или -1.
     */
    private int getRequestParameterArgumentIndex(JoinPoint joinPoint, Class<? extends Annotation> clazz) {
        MethodSignature loggedMethodSignature = (MethodSignature) joinPoint.getSignature();
        Method loggedMethod = loggedMethodSignature.getMethod();
        Annotation[][] parameterAnnotations = loggedMethod.getParameterAnnotations();
        for (int i = 0; i < parameterAnnotations.length; ++i) {
            if (Arrays.stream(parameterAnnotations[i]).anyMatch(clazz::isInstance)) {
                return i;
            }
        }
        return -1;
    }
}
