package ru.yandex.direct.intapi.validation;

import java.math.BigInteger;
import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

import ru.yandex.direct.validation.Predicates;
import ru.yandex.direct.validation.builder.Constraint;


public class IntApiConstraints {
    private IntApiConstraints() {
    }

    public static <T> Constraint<T, IntApiDefect> notNull() {
        return value -> value != null ? null : IntApiDefects.notNull();
    }

    public static <T> Constraint<T, IntApiDefect> isNull() {
        return Constraint.fromPredicateOfNullable(Objects::isNull, IntApiDefects.isNull());
    }

    public static Constraint<String, IntApiDefect> notEmptyString() {
        return Constraint.fromPredicate(Predicates.not(Predicates.empty()), IntApiDefects.notEmpty());
    }

    public static <T extends Collection> Constraint<T, IntApiDefect> notEmptyCollection() {
        return Constraint.fromPredicate(Predicates.notEmptyCollection(), IntApiDefects.notEmpty());
    }

    public static Constraint<Long, IntApiDefect> validId() {
        return Constraint.fromPredicate(Predicates.greaterThan(0L), IntApiDefects.greaterThan(0L));
    }

    public static Constraint<BigInteger, IntApiDefect> validBigIntId() {
        BigInteger border = BigInteger.ZERO;
        return Constraint.fromPredicate(Predicates.greaterThan(border), IntApiDefects.greaterThan(border));
    }

    public static Constraint<String, IntApiDefect> isLong() {
        return Constraint.fromPredicate(Predicates.isLong(), IntApiDefects.isLong());
    }

    public static Constraint<String, IntApiDefect> validDateTime(DateTimeFormatter formatter) {
        return Constraint.fromPredicate(Predicates.validDateTime(formatter), IntApiDefects.validDateTime());
    }

    public static Constraint<Long, IntApiDefect> greaterThan(Long value) {
        return Constraint.fromPredicate(Predicates.greaterThan(value), IntApiDefects.greaterThan(value));
    }

    public static Constraint<String, IntApiDefect> notNegativeWholeNumber() {
        return Constraint.fromPredicate(Predicates.isPositiveWholeNumber(), IntApiDefects.notNegativeWholeNumber());
    }

    public static Constraint<Integer, IntApiDefect> greaterThan(Integer value) {
        return Constraint.fromPredicate(Predicates.greaterThan(value), IntApiDefects.greaterThan(Long.valueOf(value)));
    }

    public static <I> Constraint<List<I>, IntApiDefect> listSize(int min, int max) {
        return Constraint.fromPredicate(Predicates.listSize(min, max), IntApiDefects.listSize(min, max));
    }
}
