package ru.yandex.direct.intapi.validation.kernel;

import java.util.Collections;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.i18n.Translatable;
import ru.yandex.direct.intapi.validation.model.IntapiError;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResponse;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResult;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.presentation.DefectPresentationRegistry;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

/**
 * Преобразование {@link ValidationResult} и {@link MassResult} в intapi ответ
 */
@Service
@ParametersAreNonnullByDefault
public class ValidationResultConversionService {

    private final PathNodeConverterProvider pathNodeConverterProvider;
    private final DefectPresentationRegistry<TranslatableIntapiDefect> defectPresentationRegistry;
    private final TranslationService translationService;

    @Autowired
    public ValidationResultConversionService(
            PathNodeConverterProvider pathNodeConverterProvider,
            DefectPresentationRegistry<TranslatableIntapiDefect> defectPresentationRegistry,
            TranslationService translationService
    ) {
        this.pathNodeConverterProvider = pathNodeConverterProvider;
        this.defectPresentationRegistry = defectPresentationRegistry;
        this.translationService = translationService;
    }

    public List<IntapiValidationResponse> buildMassValidationResponse(MassResult<?> massResult) {
        if (massResult.getResult() == null) {
            if (massResult.getValidationResult() != null) {
                return List.of(buildValidationResponse(massResult.getValidationResult()));
            } else {
                return Collections.emptyList();
            }
        }
        return StreamEx.of(massResult.getResult())
                .map(result -> ifNotNull(result, r -> buildValidationResponse(r.getValidationResult())))
                .toList();
    }

    public IntapiValidationResponse buildValidationResponse(Result<?> massResult) {
        return buildValidationResponse(massResult.getValidationResult());
    }

    public IntapiValidationResponse buildValidationResponse(
            @Nullable ValidationResult<?, Defect> validationResult) {
        return new IntapiValidationResponse(buildIntapiValidationResult(validationResult));
    }

    public IntapiValidationResult buildIntapiValidationResult(@Nullable ValidationResult<?, Defect> vr) {
        IntapiValidationResult result = new IntapiValidationResult();
        if (vr == null) {
            return result;
        }

        List<IntapiError> intapiErrors = StreamEx.of(vr.flattenErrors(pathNodeConverterProvider))
                .map(defectPresentationRegistry::getPresentation)
                .map(this::convert)
                .toList();

        result.addErrors(intapiErrors);
        return result;
    }

    public String convertErrorsToString(ValidationResult<?, Defect> vr) {
        return StreamEx.of(buildIntapiValidationResult(vr).getErrors())
                .map(IntapiError::getDescription)
                .joining(", ");
    }

    private IntapiError convert(TranslatableIntapiDefect translatableDefect) {
        Translatable translatable = translatableDefect.getTranslatable();
        DefectInfo<? extends Defect> defectInfo = translatableDefect.getDefectInfo();
        String translatedText = translationService.translate(translatable);
        return new IntapiError()
                .withPath(defectInfo.getPath().toString())
                .withCode(defectInfo.getDefect().defectId().getCode())
                .withText(translatedText)
                .withDescription(translatedText)
                .withParams(defectInfo.getDefect().params());
    }
}
