package ru.yandex.direct.intapi.webapp.semaphore;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import ru.yandex.direct.common.lettuce.LettuceConnectionProvider;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.redislock.DistributedLock;
import ru.yandex.direct.redislock.DistributedLockException;
import ru.yandex.direct.redislock.lettuce.LettuceLockBuilder;

import static ru.yandex.direct.common.configuration.RedisConfiguration.LETTUCE;

@Component
public class IntapiSemaphoreInterceptor extends HandlerInterceptorAdapter {
    static final String REQUEST_LOCK_ENTRY_NAME =
            IntapiSemaphoreInterceptor.class.getName() + ".SIMULTANEOUS_CONN_LIMIT_LOCK";
    private static final Logger LOGGER = LoggerFactory.getLogger(IntapiSemaphoreInterceptor.class);

    private final LettuceConnectionProvider lettuce;
    private final String lockKeyPrefix;

    @Autowired
    public IntapiSemaphoreInterceptor(@Qualifier(LETTUCE) LettuceConnectionProvider lettuce,
                                      @Value("${intapi-connections-lock.lock-key-prefix}") String lockKeyPrefix) {
        this.lockKeyPrefix = lockKeyPrefix;
        this.lettuce = lettuce;
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler)
            throws Exception {
        if (!(handler instanceof HandlerMethod)) {
            return true;
        }
        HandlerMethod handlerMethod = ((HandlerMethod) handler);

        Semaphore intapiSemaphore = handlerMethod.getMethod().getAnnotation(Semaphore.class);
        if (intapiSemaphore == null) {
            intapiSemaphore = handlerMethod.getBeanType().getAnnotation(Semaphore.class);
        }

        if (intapiSemaphore == null) {
            return true;
        }

        LOGGER.debug("handling semaphore for key [{}] permits [{}] retry timeout [{}ms}", intapiSemaphore.key(),
                intapiSemaphore.permits(), intapiSemaphore.retryTimeout());

        boolean locked;
        DistributedLock lock =
                createLock(intapiSemaphore.key(), intapiSemaphore.permits(), intapiSemaphore.retryTimeout(),
                        intapiSemaphore.ttl());

        try {
            locked = lock.lock();
            LOGGER.debug("semaphore for key [{}] locked. Lock object {}", intapiSemaphore.key(), lock.toString());
        } catch (Exception ex) {
            LOGGER.error("cant acquire intapi lock: lock service exception", ex);
            return true;
        }

        if (locked) {
            request.setAttribute(REQUEST_LOCK_ENTRY_NAME, lock);
            return true;
        } else {
            throw new IntApiException(HttpStatus.TOO_MANY_REQUESTS,
                    new ErrorResponse(ErrorResponse.ErrorCode.TOO_MANY_REQUESTS,
                            "Max simultaneous requests count " + intapiSemaphore.permits() + " exceeded"));
        }
    }

    DistributedLock createLock(String lockKey, int maxLocks, long lockAttemptTimeout, long ttl) {
        LettuceLockBuilder lockBuilder =
                LettuceLockBuilder.newBuilder(lettuce::getConnection).withKeyPrefix(lockKeyPrefix);
        if (ttl != 0) {
            lockBuilder = lockBuilder.withTTL(ttl);
        }
        return lockBuilder.createLock(lockKey, maxLocks, lockAttemptTimeout);
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex)
            throws Exception {
        DistributedLock lock = (DistributedLock) request.getAttribute(REQUEST_LOCK_ENTRY_NAME);
        request.removeAttribute(REQUEST_LOCK_ENTRY_NAME);
        if (lock != null) {
            try {
                lock.unlock();
                LOGGER.debug("Lock object {} unlocked", lock.toString());
            } catch (DistributedLockException lockEx) {
                LOGGER.error("cant release intapi lock: lock service exception", lockEx);
            }
        }
    }

}
