package ru.yandex.direct.intapi.webapp.semaphore.jvm;

import java.lang.reflect.Method;
import java.util.concurrent.Semaphore;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.web.core.semaphore.jvm.JvmSemaphore;
import ru.yandex.direct.web.core.semaphore.jvm.JvmSemaphoreException;

@Component
public class IntapiJvmSemaphoreInterceptor extends HandlerInterceptorAdapter {
    static final String REQUEST_SEMAPHORE_ENTRY_NAME =
            IntapiJvmSemaphoreInterceptor.class.getName() + ".SERVICE_SEMAPHORE";

    private static final Logger logger = LoggerFactory.getLogger(IntapiJvmSemaphoreInterceptor.class);

    private final JvmSemaphore semaphore;

    public IntapiJvmSemaphoreInterceptor(DirectConfig directConfig) {
        DirectConfig config = directConfig.getBranch("service_semaphore");
        this.semaphore = new JvmSemaphore(config);
    }

    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) {
        if (!(handler instanceof HandlerMethod)) {
            return true;
        }
        Method handlerMethod = ((HandlerMethod) handler).getMethod();
        String serviceName = handlerMethod.getDeclaringClass().getSimpleName();

        Semaphore sem;
        try {
            sem = semaphore.tryAcquireAndIncrementStat(serviceName);
        } catch (JvmSemaphoreException e) {
            logger.error("Can't acquire semaphore");
            throw new IntApiException(HttpStatus.TOO_MANY_REQUESTS, e.getMessage());
        }
        if (sem != null) {
            request.setAttribute(REQUEST_SEMAPHORE_ENTRY_NAME, sem);
        }
        return true;
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler,
                                Exception ex) {
        Semaphore sem = (Semaphore) request.getAttribute(REQUEST_SEMAPHORE_ENTRY_NAME);
        request.removeAttribute(REQUEST_SEMAPHORE_ENTRY_NAME);
        if (sem != null) {
            sem.release();
        }
    }
}
