package ru.yandex.direct.jobs;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.beust.jcommander.Parameter;
import com.google.common.base.MoreObjects;
import org.slf4j.Logger;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.AnnotationConfigApplicationContext;

import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.jcommander.ParserWithHelp;
import ru.yandex.direct.jobs.configuration.DebugJobRunnerConfiguration;
import ru.yandex.direct.logging.LoggingInitializer;
import ru.yandex.direct.logging.LoggingInitializerParams;
import ru.yandex.direct.scheduler.hourglass.TaskParametersMap;
import ru.yandex.direct.scheduler.support.BaseDirectJob;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.scheduler.support.ParametersSource;
import ru.yandex.direct.scheduler.support.PeriodicJobWrapper;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toList;

/**
 * Класс, позволяющий запустить одну конкретную задачу.
 * Требуется в основном для запуска при отладке.
 * Класс требуемой задачи нужно указать первым аргументом.
 * Для шардированной задачи можно указать опцию --shard X один или несколько раз,
 * задача последовательно запустится в указанных шардах.
 * Если для шардированной задачи не указать шарды - она запустится на всех шардах последовательно.
 * <p>
 * Блокировки и интерсепторы в запуске не используются.
 * <p>
 * Для параметризованных джоб нужно обязательно указазать параметр через --param
 * Пример:
 * Program arguments: CampAutoPriceQueueCleaner --shard 5 --shard 6
 * <p>
 * Если в джобе используется tvm, то нужно установить 'tvm { enabled: true }' в app-development.conf
 * и указать корректный путь до токена в 'secret'
 * <p>
 * Для запуска джобы ess-logic-processor-а (там тоже используется tvm), необходимо установить `ess { tvm { enabled:
 * true } }`
 * в direct/libs-internal/config/src/main/resources/common-development.conf
 */
public class DebugJobRunner {
    private static final Logger logger = LoggingInitializer.getLogger(DebugJobRunner.class);

    public static void main(String[] args) {
        LoggingInitializerParams loggingParams = new LoggingInitializerParams();
        RunnerParams runnerParams = new RunnerParams();
        ParserWithHelp.parse(DebugJobRunner.class.getCanonicalName(), args, loggingParams, runnerParams);
        LoggingInitializer.initialize(loggingParams);

        try (AnnotationConfigApplicationContext ctx =
                     new AnnotationConfigApplicationContext(DebugJobRunnerConfiguration.class)) {
            runJob(ctx, runnerParams);
        }
    }

    private static void runJob(AnnotationConfigApplicationContext ctx, RunnerParams runnerParams) {
        ShardHelper shardHelper = ctx.getBean(ShardHelper.class);
        runnerParams.validate(shardHelper.dbShards());

        Class<? extends BaseDirectJob> clazz = getJobClass(ctx, runnerParams.jobClassNames.get(0));
        logger.info("Found job {}", clazz.getCanonicalName());
        if (DirectShardedJob.isShardedJobClass(clazz)) {
            Collection<Integer> shards = !runnerParams.shards.isEmpty() ? runnerParams.shards : shardHelper.dbShards();
            for (int shard : shards) {
                logger.info("executing on shard {}", shard);
                PeriodicJobWrapper job = new PeriodicJobWrapper(ctx.getBean(clazz));
                job.execute(TaskParametersMap.of(DirectShardedJob.SHARD_PARAM, String.valueOf(shard)));
            }
        } else if (DirectParameterizedJob.isParameterized(clazz)) {
            ParameterizedBy parameterizedBy = clazz.getAnnotation(ParameterizedBy.class);
            Class<? extends ParametersSource> parametersSourceClazz = parameterizedBy.parametersSource();
            Set<String> allParams = new HashSet<>(DirectParameterizedJob.getAllParams(parametersSourceClazz, ctx));
            if (!runnerParams.parameters.isEmpty()) {
                for (String param : runnerParams.parameters) {
                    if (!allParams.contains(param)) {
                        logger.error(
                                "Param '{}' is not contained in all available parameters '{}', job execution will be " +
                                        "skipped",
                                param, allParams);
                        continue;
                    }
                    logger.info("executing with param {}", param);
                    PeriodicJobWrapper job = new PeriodicJobWrapper(ctx.getBean(clazz));
                    job.execute(TaskParametersMap.of(DirectParameterizedJob.PARAM_NAME, param));
                }
            }
        } else {
            PeriodicJobWrapper job = new PeriodicJobWrapper(ctx.getBean(clazz));
            job.execute(TaskParametersMap.of());
        }
    }

    /**
     * ищем в контексте Job по имени класса
     */
    private static Class<? extends BaseDirectJob> getJobClass(ApplicationContext ctx, String jobClassName) {
        Collection<BaseDirectJob> jobs = ctx.getBeansOfType(BaseDirectJob.class).values();
        List<BaseDirectJob> matchedJobs = jobs.stream()
                .filter(j ->
                        j.getClass().getCanonicalName().equals(jobClassName)
                                || j.getClass().getCanonicalName().endsWith("." + jobClassName)
                )
                .collect(toList());

        checkState(!matchedJobs.isEmpty(), "Job bean is not found: %s", jobClassName);
        checkState(matchedJobs.size() == 1, "More than one job bean found");

        return matchedJobs.get(0).getClass();
    }

    /**
     * параметры командной строки, понимаемые JobRunner
     */
    public static class RunnerParams {
        @Parameter(
                description = "Class name of job to run",
                required = true
        )
        private List<String> jobClassNames = new ArrayList<>();

        @Parameter(
                names = {"-s", "--shard"},
                description = "Shard number"
        )
        private List<Integer> shards = new ArrayList<>();

        @Parameter(
                names = {"-p", "--param"},
                description = "Parameter value"
        )
        private List<String> parameters = new ArrayList<>();

        @Override
        public String toString() {
            return MoreObjects.toStringHelper(this)
                    .add("jobClassNames", jobClassNames)
                    .add("shards", shards)
                    .add("parameters", parameters)
                    .toString();
        }

        void validate(Collection<Integer> knownShards) {
            checkArgument(jobClassNames.size() == 1, "Supported only one job class");
            Set<Integer> knownShardsSet = new HashSet<>(knownShards);
            for (Integer shard : shards) {
                checkArgument(knownShardsSet.contains(shard), "Incorrect shard: %s", shard);
            }
        }
    }
}
