package ru.yandex.direct.jobs;

import org.slf4j.Logger;
import org.springframework.context.annotation.AnnotationConfigApplicationContext;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.config.DirectConfigFactory;
import ru.yandex.direct.jcommander.ParserWithHelp;
import ru.yandex.direct.jobs.configuration.JobsConfiguration;
import ru.yandex.direct.jobs.configuration.VersionMismatchTerminator;
import ru.yandex.direct.liveresource.LiveResourceWatcher;
import ru.yandex.direct.logging.LoggingInitializer;
import ru.yandex.direct.logging.LoggingInitializerParams;
import ru.yandex.direct.scheduler.SpringAppTerminator;
import ru.yandex.direct.scheduler.hourglass.implementations.HourglassScheduler;

public final class JobsApp {
    private static final Logger logger = LoggingInitializer.getLogger(JobsApp.class);

    private JobsApp() {
        // deny installation
    }

    public static void main(String[] args) throws Exception {
        LoggingInitializerParams loggingParams = new LoggingInitializerParams();
        JobSchedulerParams schedulerParams = new JobSchedulerParams();
        ParserWithHelp.parse(JobsApp.class.getCanonicalName(), args, loggingParams, schedulerParams);
        LoggingInitializer.initialize(loggingParams, "direct.jobs");

        DirectConfig conf = DirectConfigFactory.getConfig();

        AnnotationConfigApplicationContext beanFactory = createApplicationContext(schedulerParams);

        try {
            startVersionWatcher(beanFactory);
            HourglassScheduler scheduler = beanFactory.getBean(HourglassScheduler.class);
            scheduler.saveSchedule();
            scheduler.start();
        } catch (Exception ex) {
            logger.error("failed to start scheduler", ex);
            //initializeSchedule может упасть если нет таблиц.
            //В этом случае надо остановить Spring context, иначе приложение продолжит работать(т. к. пулы потоков не
            // daemon'ы)
            SpringAppTerminator appTerminator = beanFactory.getBean(SpringAppTerminator.class);
            appTerminator.terminate(1);
        }
    }

    private static AnnotationConfigApplicationContext createApplicationContext(JobSchedulerParams schedulerParams) {
        AnnotationConfigApplicationContext beanFactory = new AnnotationConfigApplicationContext();

        beanFactory.getEnvironment().getPropertySources().addFirst(schedulerParams.asPropertySource("commandLineArgs"));

        beanFactory.register(JobsConfiguration.class);
        //Web-приложения закрывают контекст в FrameworkServlet.destroy(), Standalone - через
        // hook/ConfigurableApplicationContext.close()
        beanFactory.registerShutdownHook();
        beanFactory.refresh();

        return beanFactory;
    }

    private static void startVersionWatcher(AnnotationConfigApplicationContext beanFactory) {
        VersionMismatchTerminator versionMismatchTerminator = beanFactory.getBean(VersionMismatchTerminator.class);
        LiveResourceWatcher versionFileWatcher =
                beanFactory.getBean("schedulerVersionWatcher", LiveResourceWatcher.class);
        logger.info("using {} file to track actual app version", versionFileWatcher.getLiveResource().getLocation());
        String newVersion = versionFileWatcher.getLiveResource().getContent();
        //эту проверка будет производиться слушателем обновлений файла (LiveResourceWatcher.addListener), но между
        // запуском приложения и
        //созданием Spring контекста файл с версией может обновится. Такое обновление не сгенерирует event, поэтому
        // надо проверить явно,
        //уже после запуска LiveResourceWatcher
        versionMismatchTerminator.terminateIfVersionMismatch(newVersion);
    }

}
