package ru.yandex.direct.jobs;

import java.util.ArrayList;
import java.util.List;

import com.beust.jcommander.Parameter;
import com.google.common.base.MoreObjects;
import org.slf4j.Logger;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.AnnotationConfigApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.direct.hourglass.RandomChooser;
import ru.yandex.direct.hourglass.SchedulerInstancePinger;
import ru.yandex.direct.hourglass.implementations.SimpleSchedulerInstancePinger;
import ru.yandex.direct.hourglass.implementations.randomchoosers.SimpleRandomChooserImpl;
import ru.yandex.direct.hourglass.implementations.updateschedule.SimpleScheduleUpdateServiceImpl;
import ru.yandex.direct.hourglass.storage.PrimaryId;
import ru.yandex.direct.hourglass.storage.Storage;
import ru.yandex.direct.hourglass.storage.implementations.memory.MemStorageImpl;
import ru.yandex.direct.hourglass.updateschedule.ScheduleUpdateService;
import ru.yandex.direct.jcommander.ParserWithHelp;
import ru.yandex.direct.jobs.configuration.HourglassConfiguration;
import ru.yandex.direct.jobs.configuration.JobsConfiguration;
import ru.yandex.direct.logging.LoggingInitializer;
import ru.yandex.direct.logging.LoggingInitializerParams;
import ru.yandex.direct.scheduler.SpringAppTerminator;
import ru.yandex.direct.scheduler.hourglass.TaskListProvider;
import ru.yandex.direct.scheduler.hourglass.implementations.FilteredTaskListProvider;
import ru.yandex.direct.scheduler.hourglass.implementations.HourglassScheduler;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.configuration.DirectHourglassConfiguration.VERSION_BEAN_NAME;

public final class SingleHourglassJobRunner {
    private static final Logger logger = LoggingInitializer.getLogger(JobsApp.class);

    private static String jobClass;

    private SingleHourglassJobRunner() {
    }

    public static void main(String[] args) {
        LoggingInitializerParams loggingParams = new LoggingInitializerParams();

        RunnerParams runnerParams = new RunnerParams();
        ParserWithHelp.parse(JobsApp.class.getCanonicalName(), args, loggingParams, runnerParams);
        runnerParams.validate();

        jobClass = runnerParams.jobClassNames.get(0);

        LoggingInitializer.initialize(loggingParams, "direct.jobs");

        AnnotationConfigApplicationContext beanFactory = createApplicationContext();
        HourglassScheduler scheduler = beanFactory.getBean(HourglassScheduler.class);

        try {
            scheduler.saveSchedule();
            scheduler.start();
        } catch (Exception ex) {
            logger.error("failed to start scheduler", ex);
            //initializeSchedule может упасть если нет таблиц.
            //В этом случае надо остановить Spring context, иначе приложение продолжит работать(т. к. пулы потоков не
            // daemon'ы)
            SpringAppTerminator appTerminator = beanFactory.getBean(SpringAppTerminator.class);
            appTerminator.terminate(1);
        }
    }

    private static AnnotationConfigApplicationContext createApplicationContext() {
        AnnotationConfigApplicationContext beanFactory = new AnnotationConfigApplicationContext();
        beanFactory.register(SingleJobRunnerConfiguration.class);
        //Web-приложения закрывают контекст в FrameworkServlet.destroy(), Standalone - через
        // hook/ConfigurableApplicationContext.close()
        beanFactory.registerShutdownHook();
        beanFactory.refresh();
        return beanFactory;
    }

    public static class RunnerParams {
        @Parameter(
                description = "Class name of job to run",
                required = true
        )
        private List<String> jobClassNames = new ArrayList<>();

        @Override
        public String toString() {
            return MoreObjects.toStringHelper(this)
                    .add("jobClassNames", jobClassNames)
                    .toString();
        }

        void validate() {
            checkArgument(jobClassNames.size() == 1, "Supported only one job class");
        }
    }

    @Configuration
    @Import({
            HourglassConfiguration.class,
            JobsConfiguration.class,
    })
    public static class SingleJobRunnerConfiguration {
        @Bean
        public Storage storage() {
            return new MemStorageImpl();
        }

        @Bean
        public ScheduleUpdateService scheduleUpdateService(Storage storage) {
            return new SimpleScheduleUpdateServiceImpl(storage);
        }

        @Bean
        public TaskListProvider taskListProvider(ApplicationContext context) {
            return new FilteredTaskListProvider(context, jobClass);
        }

        @Bean
        public RandomChooser<PrimaryId> randomChooser() {
            return new SimpleRandomChooserImpl<>();
        }

        @Bean(VERSION_BEAN_NAME)
        public String version() {
            return "0.0-0";
        }

        @Bean
        public SchedulerInstancePinger schedulerInstancePinger() {
            return new SimpleSchedulerInstancePinger();
        }

    }
}
