package ru.yandex.direct.jobs.abt;

import java.net.URI;
import java.time.Clock;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.abt.check.TableExistsChecker;
import ru.yandex.direct.jobs.abt.queryconf.AbPrepareQueryConf;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;

import static java.time.temporal.ChronoUnit.DAYS;
import static ru.yandex.direct.common.db.PpcPropertyNames.lastAbPreparedDate;
import static ru.yandex.direct.jobs.abt.AbUtils.formatInstantAsDate;
import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;

/**
 * Джоба для подговки данных для расчета AB - метрик в cofe (https://wiki.yandex-team.ru/serp/experiments/cofe/doc/)
 * Запускает yql для подготовки. Чтобы добавить новый yql нужно добавить сам yql, а так же кофигурацию, реализацию
 * {@link AbPrepareQueryConf}
 * Запрос запускается за даты с начала прошлого успешного запуска и по предыдущий день
 * Запускается только если итоговой таблицы запроса еще нет
 *
 * Локально запустить из DebugJobRunner можно с такими параметрами, например:
 * AbDataPrepareJob --param "HAHN---prepare_binlog_events.sql"
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 1),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_1})
@Hourglass(periodInSeconds = 3600, needSchedule = ProductionOnly.class)
@ParameterizedBy(parametersSource = AbDataPrepareParametersSource.class)
public class AbDataPrepareJob extends DirectParameterizedJob<DirectParameterizedJob> {

    private static final Logger logger = LoggerFactory.getLogger(AbDataPrepareJob.class);

    private final YtProvider ytProvider;
    private final TableExistsChecker tableExistsChecker;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final String resultRelativePath;
    private final String queriesFolder;
    private final String ytPool;
    private final AbDataPrepareParametersSource abDataPrepareParametersSource;

    private final Clock clock;

    @Autowired
    public AbDataPrepareJob(YtProvider ytProvider, TableExistsChecker tableExistsChecker,
                            PpcPropertiesSupport ppcPropertiesSupport,
                            AbDataPrepareParametersSource abDataPrepareParametersSource,
                            @Value("${abt.prepare.result_relative_path}") String resultRelativePath,
                            @Value("${abt.prepare.queries_folder}") String queriesFolder,
                            @Value("${abt.prepare.yt_pool}") String ytPool) {
        this(ytProvider, tableExistsChecker, ppcPropertiesSupport, abDataPrepareParametersSource, resultRelativePath,
                queriesFolder, ytPool, Clock.systemUTC());
    }

    public AbDataPrepareJob(YtProvider ytProvider, TableExistsChecker tableExistsChecker,
                            PpcPropertiesSupport ppcPropertiesSupport,
                            AbDataPrepareParametersSource abDataPrepareParametersSource,
                            String resultRelativePath,
                            String queriesFolder,
                            String ytPool,
                            Clock clock) {
        this.ytProvider = ytProvider;
        this.tableExistsChecker = tableExistsChecker;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.abDataPrepareParametersSource = abDataPrepareParametersSource;
        this.resultRelativePath = resultRelativePath;
        this.queriesFolder = queriesFolder;
        this.ytPool = ytPool;
        this.clock = clock;
    }

    @Override
    public void execute() {
        var paramString = getParam();
        var abDataPrepareParameter = abDataPrepareParametersSource.convertStringToParam(paramString);
        var lastAbPrepareDateProperty = ppcPropertiesSupport.get(lastAbPreparedDate(paramString));
        var lastCalculatedDate = lastAbPrepareDateProperty.get();

        var calculateDateRange = getCalculateDatesRange(lastCalculatedDate);
        var startDate = calculateDateRange.getLeft();
        var finishDate = calculateDateRange.getRight();
        logger.info("Try prepare data between {} and {}", startDate, finishDate);

        Instant lastSuccessfulDate = null;
        for (var calculateDate = startDate; calculateDate.compareTo(finishDate) <= 0; calculateDate =
                calculateDate.plus(1, DAYS)) {

            var isSuccessful = executeForDate(calculateDate, abDataPrepareParameter);
            if (!isSuccessful) {
                logger.info("Can't prepare data for query {} for date {}",
                        abDataPrepareParameter.getQueryConfiguration().getQueryPath(), calculateDate);
                break;
            } else {
                lastSuccessfulDate = calculateDate;
            }
        }
        if (lastSuccessfulDate != null) {
            lastAbPrepareDateProperty.set(lastSuccessfulDate.getEpochSecond());
        }
    }

    /**
     * Выполняет запуск yql для конкретной даты
     * Запускает только если выполняется условия для заупска запроса
     * {@link AbPrepareQueryConf#isReady(AbPreparedContext)}
     *
     * @return true если результат готов, false, если нет
     */
    private boolean executeForDate(Instant calculateDate, AbDataPrepareParameter param) {
        var clusterYtConfig = ytProvider.getClusterConfig(param.getYtCluster());
        var operator = ytProvider.getOperator(param.getYtCluster(), YtSQLSyntaxVersion.SQLv1);
        var homePath = clusterYtConfig.getHome();
        var calculateDateFormatted = formatInstantAsDate(calculateDate);
        var resultPath = YtPathUtil.generatePath(homePath, relativePart(), resultRelativePath, calculateDateFormatted,
                param.getQueryConfiguration().getDestTable());

        // Если таблица за расчитываемую дату уже есть, значит ничего считать не надо
        if (tableExistsChecker.check(param.getYtCluster(), resultPath)) {
            logger.info("Result table {} already exists", resultPath);
            return true;
        }

        var abPreparedContext = new AbPreparedContext(calculateDate, param.getYtCluster());

        var isReadyToCalculate = param.getQueryConfiguration().isReady(abPreparedContext);
        var queryPath = param.getQueryConfiguration().getQueryPath();
        if (!isReadyToCalculate) {
            logger.info("Query {} doesnt't ready to execute", queryPath);
            return false;
        }
        URI uri = URI.create(queriesFolder).resolve(queryPath);
        var query = LiveResourceFactory.get(uri.toString()).getContent();
        List<Object> queryParams = new ArrayList<>();
        String currentYtPool;
        if (ytPool.isEmpty()) {
            // если пул не задан в настройках - используем пул-имени-пользователя
            currentYtPool = clusterYtConfig.getUser();
        } else {
            currentYtPool = ytPool;
        }
        queryParams.add(currentYtPool);
        queryParams.add(calculateDateFormatted);
        queryParams.add(resultPath);
        queryParams.addAll(param.getQueryConfiguration().getQueryAdditionalParams());
        operator.yqlExecute(query, queryParams.toArray());
        return true;
    }

    /**
     * Возвращает начальную и конечную дату расчета
     * Конечная дата - всегда прошлый день
     * Начальная дата - либо lastCalculatedDate + день, либо прошлый день
     *
     * @return пара начало периода - конец периода
     */
    Pair<Instant, Instant> getCalculateDatesRange(Long lastCalculatedDate) {
        var finishDate = clock.instant().truncatedTo(DAYS).minus(1, DAYS);
        Instant startDate;
        if (lastCalculatedDate == null) {
            startDate = finishDate;
        } else {
            startDate = Instant.ofEpochSecond(lastCalculatedDate).plus(1, DAYS).truncatedTo(DAYS);
        }
        return Pair.of(startDate, finishDate);
    }
}
