package ru.yandex.direct.jobs.accountscore;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.configuration.DirectExportYtClustersParametersSource;
import ru.yandex.direct.jobs.util.yql.CommonYqlExport;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Расчёт в YT данных по каждому клиенту.
 * Используется для вычисления показателя качества аккаунта.
 * <p>
 * NB: способ расчета факторов и их веса должны совпадать с perl-реализацией!
 * https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/Direct/YT/account_score_yt.pm
 */
@Hourglass(periodInSeconds = 5400, needSchedule = ProductionOnly.class)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 7),
        tags = {DIRECT_PRIORITY_2, CheckTag.YT, CheckTag.GROUP_INTERNAL_SYSTEMS},
        needCheck = ProductionOnly.class)
@ParameterizedBy(parametersSource = DirectExportYtClustersParametersSource.class)
@ParametersAreNonnullByDefault
public class CalcFactorsJob extends DirectParameterizedJob<YtCluster> {
    private static final Logger logger = LoggerFactory.getLogger(CalcFactorsJob.class);

    private static final String QUERY_RESOURCE = "classpath:///accountscore/CalcAccountScoreFactors.yql";
    private static final String EXPORT_PATH = "import/account_score/accounts_data";
    /**
     * За сколько дней считать данные о днях простоя.
     * Должно совпадать со значением в perl.
     * https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/Direct/YT/account_score_yt.pm?rev=4927484#L35
     */
    private static final int STOP_DAYS_STAT_LENGTH = 30;

    private final CommonYqlExport.Builder exportBuilder;
    private final DirectExportYtClustersParametersSource parametersSource;

    @Autowired
    public CalcFactorsJob(DirectExportYtClustersParametersSource parametersSource, YtProvider ytProvider) {
        this.parametersSource = parametersSource;

        exportBuilder = new CommonYqlExport.Builder(logger, ytProvider, QUERY_RESOURCE, EXPORT_PATH)
                .withBindingsProvider(CalcFactorsJob::getBindings);
    }

    @Override
    public void execute() {
        YtCluster ytCluster = parametersSource.convertStringToParam(getParam());
        exportBuilder.build(ytCluster).generateIfNeeded();
    }

    private static Object[] getBindings(CommonYqlExport.Context ctx) {
        return new Object[]{
                ctx.getDestinationTablePathAsString(),
                STOP_DAYS_STAT_LENGTH
        };
    }
}
