package ru.yandex.direct.jobs.adfox.messaging;

import java.time.LocalDateTime;
import java.time.format.DateTimeParseException;
import java.util.Collection;
import java.util.Optional;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.common.db.PpcPropertyParseException;
import ru.yandex.direct.common.db.PpcPropertyType;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.jobs.adfox.messaging.ytutils.YtOrderedTableReader;
import ru.yandex.direct.jobs.adfox.messaging.ytutils.YtReadException;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.yt.ytclient.proxy.YtClient;

import static ru.yandex.direct.common.db.PpcPropertyNames.ADFOX_INPUT_PROCESSING_ENABLED;
import static ru.yandex.direct.jobs.adfox.messaging.AdfoxMessagingProperties.createAdfoxMessagingProperties;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

/**
 * Job для обработки входящих сообщений от Adfox. Сообщения бывают разных типов.
 * Обработчики сообщений реализуют интерфейс {@link AdfoxMessageHandler}.
 * <p>
 * В случае ошибки зажигает мониторинг (WARN). Запоминает время последней ошибки
 * и продолжает сигналить, пока не будет погашен.
 * <p>
 * Чтобы погасить, нужно установить новое значение ppc property {@link PpcPropertyKey#ADFOX_INPUT_ERR_REVIEWED} –
 * дату, когда была просмотрена последняя ошибка. Дата пишется в формате локального времени ISO_LOCAL_DATE_TIME,
 * пример: {@code 2018-12-31T15:59:59}. В случае ошибки в формате даты также зажигается статус WARN.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 20),
        needCheck = NonDevelopmentEnvironment.class,
        tags = {DIRECT_PRIORITY_2, CheckTag.JOBS_RELEASE_REGRESSION},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.CHAT_API_MONITORING,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}
        )
)
@Hourglass(periodInSeconds = 60, needSchedule = NonDevelopmentEnvironment.class)
@ParametersAreNonnullByDefault
public class ProcessIncomingAdfoxMessagesJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(ProcessIncomingAdfoxMessagesJob.class);

    /**
     * Ключи PPC_PROPERTIES.
     */
    static class PpcPropertyKey {
        /**
         * Ключ проперти: дата и время последней ошибки при обработке сообщения из adFox
         */
        static final PpcPropertyName<LocalDateTime> ADFOX_INPUT_ERR_LAST =
                new PpcPropertyName<>("adfox.input.err.last", PpcPropertyType.LOCAL_DATE_TIME);
        /**
         * Ключ проперти: дата и время последнего ревью ошибок из adFox человеком
         */
        static final PpcPropertyName<LocalDateTime> ADFOX_INPUT_ERR_REVIEWED =
                new PpcPropertyName<>("adfox.input.err.reviewed", PpcPropertyType.LOCAL_DATE_TIME);
        /**
         * Ключ, по которому хранится offset последнего прочитанного и обработанного сообщения
         */
        private static final PpcPropertyName<Long> ADFOX_INPUT_QUEUE_OFFSET_KEY =
                new PpcPropertyName<>("adfox.input.queue.offset", PpcPropertyType.LONG);
    }

    private final YtProvider ytProvider;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final AdfoxMessagingProperties messagingProperties;
    private final RawMessageConsumer rawMessageConsumer;

    @Autowired
    public ProcessIncomingAdfoxMessagesJob(
            DirectConfig directConfig,
            Collection<AdfoxMessageHandler> adfoxMessageHandlers,
            YtProvider ytProvider,
            PpcPropertiesSupport ppcPropertiesSupport
    ) {
        this.ytProvider = ytProvider;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        messagingProperties = createAdfoxMessagingProperties(directConfig);
        rawMessageConsumer = new RawMessageConsumer(listToMap(adfoxMessageHandlers, AdfoxMessageHandler::getType));
    }

    @Override
    public void execute() {
        boolean jobEnabled = ppcPropertiesSupport.get(ADFOX_INPUT_PROCESSING_ENABLED).getOrDefault(false);
        if (!jobEnabled) {
            logger.info("Skip processing. Job is not enabled. {}={}", ADFOX_INPUT_PROCESSING_ENABLED.getName(),
                    jobEnabled);
            return;
        }

        QueueProcessor.Result processingResult = runRpcCommand(this::processMessagesFromQueue);
        registerErrors(processingResult);
        notifyJugglerIfNeeded(processingResult);
    }

    /**
     * Проверяет результат на наличие ошибок и выставляет в базе флаг для мониторига.
     *
     * @param processingResult результат
     */
    private void registerErrors(QueueProcessor.Result processingResult) {
        if (processingResult == QueueProcessor.Result.ERRORS) {
            ppcPropertiesSupport.get(PpcPropertyKey.ADFOX_INPUT_ERR_LAST).set(LocalDateTime.now());
        }
    }

    <T> T runRpcCommand(Function<YtClient, T> command) {
        return ytProvider.getDynamicOperator(getYtCluster()).runRpcCommand(command);
    }

    private YtCluster getYtCluster() {
        return messagingProperties.inputQueue().cluster();
    }

    /**
     * @param client клиент YT
     * @return Результат обработки очереди, {@link QueueProcessor.Result}.
     */
    private QueueProcessor.Result processMessagesFromQueue(YtClient client) throws YtReadException {
        PpcProperty<Long> offsetProperty = ppcPropertiesSupport.get(PpcPropertyKey.ADFOX_INPUT_QUEUE_OFFSET_KEY);
        long offset = offsetProperty.getOrDefault(-1L);

        logger.info("Running message processing from {} by offset {}", messagingProperties.inputQueue(), offset);

        YtOrderedTableReader tableReader = getYtOrderedTableReader(client);
        QueueProcessor queueProcessor = new QueueProcessor(tableReader, rawMessageConsumer);
        QueueProcessor.Result result;
        try {
            result = queueProcessor.processNewMessages(offset);
        } catch (YtReadException e) {
            // Yt read exception, re-throw
            throw e;
        } catch (RuntimeException e) {
            logger.error("Error when processing message queue {}", messagingProperties.inputQueue(), e);
            result = QueueProcessor.Result.ERRORS;
        }
        long lastSuccessfulRowIndex = queueProcessor.getLastSuccessfulRowIndex();
        offsetProperty.set(lastSuccessfulRowIndex);
        return result;
    }

    private YtOrderedTableReader getYtOrderedTableReader(YtClient client) {
        String queuePath = messagingProperties.inputQueue().path();
        return new YtOrderedTableReader(queuePath, client);
    }

    private void notifyJugglerIfNeeded(QueueProcessor.Result processingResult) {
        if (processingResult == QueueProcessor.Result.ERRORS) {
            setJugglerStatus(JugglerStatus.WARN, "Got new errors during Adfox incoming messages processing");
            return;
        }
        try {
            if (notReviewedErrorsPresent()) {
                setJugglerStatus(JugglerStatus.WARN, "There are still Adfox messaging errors waiting for the review");
            }
        } catch (PpcPropertyParseException ex) {
            setJugglerStatus(JugglerStatus.WARN, "Unable to parse date from ppc property");
        }
    }

    /**
     * @return {@code true} если есть необработанные ошибки, иначе {@code false}.
     * @throws DateTimeParseException если не удалось прочитать даты из базы; скорее всего в таблице опечатка.
     */
    private boolean notReviewedErrorsPresent() {
        Optional<LocalDateTime> lastErrorOccurred = getDateTimeProp(PpcPropertyKey.ADFOX_INPUT_ERR_LAST);
        if (!lastErrorOccurred.isPresent()) {
            return false;
        }
        Optional<LocalDateTime> lastErrorReviewed = getDateTimeProp(PpcPropertyKey.ADFOX_INPUT_ERR_REVIEWED);
        return lastErrorReviewed.map(reviewed -> lastErrorOccurred.get().isAfter(reviewed)).orElse(true);
    }

    private Optional<LocalDateTime> getDateTimeProp(PpcPropertyName<LocalDateTime> propName) {
        return Optional.of(propName).map(ppcPropertiesSupport::get).map(PpcProperty::get);
    }
}
