package ru.yandex.direct.jobs.adfox.messaging;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.jobs.adfox.messaging.ytutils.RawQueueMessage;
import ru.yandex.direct.jobs.adfox.messaging.ytutils.RawQueueMessageRecordParser;
import ru.yandex.direct.jobs.adfox.messaging.ytutils.YtOrderedTableReader;
import ru.yandex.direct.jobs.adfox.messaging.ytutils.YtReadException;

/**
 * Обработчик сообщений из очереди.
 * <p>
 * Вычитка осуществляется {@link YtOrderedTableReader}.
 * Обработка отдельных сообщений делегируется в {@link RawMessageConsumer}.
 */
class QueueProcessor {

    /**
     * Результат обработки очереди.
     */
    enum Result {
        /**
         * Не было ошибок.
         */
        SUCCESS,
        /**
         * Была по крайней мере одна ошибка.
         */
        ERRORS,
    }

    private static final Logger logger = LoggerFactory.getLogger(QueueProcessor.class);

    // С этим tag'ом будут логироваться сообщения о неуспешных обаботках сообщений. См. DIRECT-72119
    private static final Logger RECOVERY_LOGGER = LoggerFactory.getLogger("TRANSPORT.ADFOX.recovery");
    /**
     * Количество вычитываемых за раз сообщений из очереди
     */
    private static final int READ_BATCH_SIZE = 10;

    private final YtOrderedTableReader tableReader;
    private final RawMessageConsumer rawMessageConsumer;
    private long lastSuccessfulRowIndex = -1;

    QueueProcessor(YtOrderedTableReader tableReader, RawMessageConsumer rawMessageConsumer) {
        this.tableReader = tableReader;
        this.rawMessageConsumer = rawMessageConsumer;
    }

    /**
     * @param offset {@code $row_index}, с которого необходимо считывать новые сообщения
     */
    Result processNewMessages(long offset) throws YtReadException {
        Result result = Result.SUCCESS;
        lastSuccessfulRowIndex = offset;
        RawQueueMessageRecordParser rawQueueMessageRecordParser = RawQueueMessageRecordParser.getInstance();
        int batchSize = READ_BATCH_SIZE;
        List<RawQueueMessage> messages;
        do {
            messages = tableReader.read(lastSuccessfulRowIndex, batchSize, rawQueueMessageRecordParser);
            logger.debug("Read {} messages. Processing...", messages.size());

            for (RawQueueMessage message : messages) {
                try {
                    logger.debug("Try process {}", message);
                    rawMessageConsumer.consume(message);
                } catch (Exception e) {
                    // ошибка обработки отдельного сообщения не прерывает обработку всей очереди
                    // логируем ошибку и продолжаем
                    logger.error("Error on parsing message ($row_index: {}): {}",
                            message.getRowIndex(), e.getMessage(), e);
                    RECOVERY_LOGGER.warn("Error \"{}\" on parsing message ($row_index: {}): {}",
                            e.getMessage(), message.getRowIndex(), message.getMessage());
                    result = Result.ERRORS;
                }
                lastSuccessfulRowIndex = message.getRowIndex();
            }
        } while (messages.size() >= batchSize);

        logger.info("Successfully process rows ({}, {}]", offset, lastSuccessfulRowIndex);
        return result;
    }

    /**
     * @return {@code $row_index} последнего успешно прочитанного сообщения
     */
    long getLastSuccessfulRowIndex() {
        return lastSuccessfulRowIndex;
    }
}
