package ru.yandex.direct.jobs.adfox.messaging.export;

import java.time.Duration;
import java.util.Collection;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.deal.model.DealDirect;
import ru.yandex.direct.core.entity.deal.model.DealSimple;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.jobs.adfox.messaging.AdfoxMessagingProperties;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtDynamicOperator;

import static ru.yandex.direct.common.db.PpcPropertyNames.ADFOX_OUTPUT_ENABLED;
import static ru.yandex.direct.jobs.adfox.messaging.AdfoxMessagingProperties.createAdfoxMessagingProperties;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.NotificationRecipient.LOGIN_PAVRYABOV;

@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 20),
        needCheck = NonDevelopmentEnvironment.class,
        tags = {DIRECT_PRIORITY_2, CheckTag.JOBS_RELEASE_REGRESSION},
        notifications = @OnChangeNotification(
                recipient = LOGIN_PAVRYABOV,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@Hourglass(periodInSeconds = 60, needSchedule = NonDevelopmentEnvironment.class)
public class AdfoxDealExportJob extends DirectShardedJob {

    private static final Logger logger = LoggerFactory.getLogger(AdfoxDealExportJob.class);

    /**
     * максимальное количество сделок отправляемых за один запуск job'ы
     */
    private static final int SYNC_BATCH_SIZE = 1000;
    /**
     * таймаут на операцию добавления сообщений об изменении сделок в YT-таблицу
     */
    private static final Duration YT_OPERATION_TIMEOUT = Duration.ofSeconds(10L);

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final AdfoxDealExportService adfoxDealExportService;
    private final YtProvider ytProvider;
    private final AdfoxMessagingProperties messagingProperties;

    @Autowired
    public AdfoxDealExportJob(
            PpcPropertiesSupport ppcPropertiesSupport,
            DirectConfig directConfig,
            AdfoxDealExportService adfoxDealExportService,
            YtProvider ytProvider)
    {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.adfoxDealExportService = adfoxDealExportService;
        this.ytProvider = ytProvider;
        messagingProperties = createAdfoxMessagingProperties(directConfig);
    }

    @Override
    public void execute() {
        PpcProperty<Boolean> property = ppcPropertiesSupport.get(ADFOX_OUTPUT_ENABLED);
        Boolean jobEnabled = property.find().orElse(false);
        if (!jobEnabled) {
            logger.info("Skip processing. Job is not enabled. {}={}", ADFOX_OUTPUT_ENABLED, property.get());
            return;
        }

        int shard = getShard();
        List<DealDirect> dealsToSync = adfoxDealExportService.pickDealsToSync(shard, SYNC_BATCH_SIZE);

        if (dealsToSync.isEmpty()) {
            logger.info("No deals to sync in shard {}", shard);
            return;
        }

        logger.debug("Creating transaction for adding info about {} deals", dealsToSync.size());
        YtDynamicOperator dynamicOperator = ytProvider.getDynamicOperator(messagingProperties.outputQueue().cluster());
        DealUpdateMessageSender sender = new DealUpdateMessageSender(messagingProperties.outputQueue().path());
        dynamicOperator.runInTransaction(sender.makeSenderFor(dealsToSync));
        Collection<DealSimple> dealSynced = sender.getSyncedDeals();

        adfoxDealExportService.markDealsSynced(shard, dealSynced);

        logger.info("Successfully synced {} deals", dealsToSync.size());
    }

}
