package ru.yandex.direct.jobs.adfox.messaging.export;

import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.function.Predicate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.deal.model.DealDirect;
import ru.yandex.direct.core.entity.deal.model.DealSimple;
import ru.yandex.direct.core.entity.deal.model.StatusAdfoxSync;
import ru.yandex.direct.core.entity.deal.repository.DealRepository;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Вспомогательный сервис для {@link AdfoxDealExportJob}.
 * Умеет отбирать сделки для отправки в Adfox и обновлять у них статусы синхронизации.
 */
@Service
public class AdfoxDealExportService {
    private static final Logger logger = LoggerFactory.getLogger(AdfoxDealExportService.class);

    /**
     * статусы сделок, означающие, что их необхоидмо отправить в Adfox
     */
    private static final EnumSet<StatusAdfoxSync> NOT_SYNCED_STATUSES =
            EnumSet.of(StatusAdfoxSync.NO, StatusAdfoxSync.SENDING);

    private final DealRepository dealRepository;

    @Autowired
    public AdfoxDealExportService(DealRepository dealRepository) {
        this.dealRepository = dealRepository;
    }

    /**
     * Отбирает из {@code shard} шарда {@code limit} сделок со статусом
     * {@link ru.yandex.direct.dbschema.ppc.tables.Deals#STATUS_ADFOX_SYNC} {@code in ('No', 'Sending')}
     * (см. {@link #NOT_SYNCED_STATUSES})
     * и переводит их статусы в {@code 'Sending'}
     */
    public List<DealDirect> pickDealsToSync(int shard, int limit) {
        List<DealDirect> dealsToSync =
                dealRepository.getDirectDealsByAdfoxSyncStatus(shard, NOT_SYNCED_STATUSES, LimitOffset.limited(limit));

        logger.debug("Selected {} deals to send", dealsToSync.size());
        if (dealsToSync.isEmpty()) {
            logger.info("No deals to sync. Run completed");
            return emptyList();
        }

        Predicate<DealDirect> dealHasStatusSyncNo = deal -> deal.getStatusAdfoxSync() == StatusAdfoxSync.NO;
        List<DealDirect> dealWithStatusSyncedNo = filterList(dealsToSync, dealHasStatusSyncNo);
        logger.debug("Update statusAdfoxSynced to 'Sending' for {} deals", dealWithStatusSyncedNo.size());
        dealRepository.updateAdfoxSyncStatus(shard, mapList(dealWithStatusSyncedNo, ModelWithId::getId),
                StatusAdfoxSync.NO, StatusAdfoxSync.SENDING);
        // Приведём статус в актуальное состояние
        dealsToSync.forEach(deal -> deal.setStatusAdfoxSync(StatusAdfoxSync.SENDING));
        return dealsToSync;
    }

    /**
     * Для переданных сделок {@code dealSynced} обновляет значение
     * {@link ru.yandex.direct.dbschema.ppc.tables.Deals#STATUS_ADFOX_SYNC} на {@code 'Yes'},
     * если оно было в значении {@code 'Sending'}.
     */
    public void markDealsSynced(int shard, Collection<? extends DealSimple> dealSynced) {
        logger.debug("Update statusAdfoxSynced to 'Yes' for {} deals", dealSynced.size());
        int statusUpdatedCount = dealRepository.updateAdfoxSyncStatus(shard, mapList(dealSynced, ModelWithId::getId),
                StatusAdfoxSync.SENDING, StatusAdfoxSync.YES);
        if (statusUpdatedCount != dealSynced.size()) {
            logger.info("{} deals of {} are marked as synced", statusUpdatedCount, dealSynced.size());
        }
    }
}
