package ru.yandex.direct.jobs.adfox.messaging.export;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.function.Consumer;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.adfoxmessaging.AdfoxMessagingUtils;
import ru.yandex.direct.adfoxmessaging.protos.CommonMessage;
import ru.yandex.direct.core.entity.deal.model.DealDirectSync;
import ru.yandex.direct.core.entity.deal.model.DealSimple;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;

import static java.util.Arrays.asList;
import static java.util.Collections.unmodifiableCollection;

/**
 * Класс занимается отправкой информации об изменённых сделках в очередь исходящих сообщений в Adfox на YT.
 */
@ParametersAreNonnullByDefault
class DealUpdateMessageSender {
    private static final Logger logger = LoggerFactory.getLogger(DealUpdateMessageSender.class);

    private static final TableSchema EVENTS_TABLE_SCHEMA = new TableSchema.Builder()
            // без ключевой колонки создать TableSchema нельзя, даже если она не используется
            .addKey("$tablet_index", ColumnValueType.INT64)
            .addValue("event_json", ColumnValueType.STRING)
            .build();

    private final String path;
    private final DealToUpdateMessageConverter dealToUpdateMessageConverter;
    private final Collection<DealSimple> syncedDeals;

    /**
     * @param adfoxOutputQueuePath — путь до очереди исходящих сообщений в Adfox
     */
    DealUpdateMessageSender(String adfoxOutputQueuePath) {
        this.path = adfoxOutputQueuePath;
        dealToUpdateMessageConverter = new DealToUpdateMessageConverter();
        syncedDeals = new ArrayList<>();
    }

    /**
     * Возвращает {@link Consumer}&lt;{@link ApiServiceTransaction}&gt;,
     * который выполняет операцию по отправке изменённых сделок {@code deals} в YT
     */
    Consumer<ApiServiceTransaction> makeSenderFor(Collection<? extends DealDirectSync> deals) {
        return transaction -> sendDeals(transaction, deals);
    }

    /**
     * Отправляет переданные сделки {@code deals} в YT с помощью {@code transaction}
     *
     * @return нбор сделок, которые были успешно отправлены.
     */
    <T extends DealDirectSync> List<T> sendDeals(ApiServiceTransaction transaction, Collection<T> deals) {
        Instant currentTime = Instant.now();
        List<List<Object>> rowsToInsert = new ArrayList<>();
        List<T> updatedDeals = new ArrayList<>();
        for (T deal : deals) {
            try {
                CommonMessage updateDealMessage = dealToUpdateMessageConverter.dealToUpdateMessage(deal, currentTime);
                String json = AdfoxMessagingUtils.toJson(updateDealMessage);
                rowsToInsert.add(asList(null /*$tablet_index*/, json));
                updatedDeals.add(deal);
            } catch (Exception e) {
                logger.warn("Error during deal {}", deal, e);
            }
        }

        logger.debug("Adding {} rows to YT table '{}'", rowsToInsert.size(), path);
        ModifyRowsRequest modifyRowsRequest = new ModifyRowsRequest(path, EVENTS_TABLE_SCHEMA).addInserts(rowsToInsert);
        transaction.modifyRows(modifyRowsRequest).join(); // IGNORE-BAD-JOIN DIRECT-149116

        syncedDeals.addAll(updatedDeals);
        return updatedDeals;
    }

    /**
     * @return набор сделок, которые были успешно отправлены в Adfox
     */
    Collection<DealSimple> getSyncedDeals() {
        return unmodifiableCollection(syncedDeals);
    }
}
