package ru.yandex.direct.jobs.adfox.messaging.handler;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.adfoxmessaging.protos.AdfoxDealCreatePayload;
import ru.yandex.direct.adfoxmessaging.protos.AdfoxDealStatus;
import ru.yandex.direct.core.entity.deal.model.Deal;
import ru.yandex.direct.core.entity.deal.model.DealPlacement;
import ru.yandex.direct.core.entity.deal.model.DealType;
import ru.yandex.direct.core.entity.deal.model.StatusAdfox;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.misc.lang.StringUtils;

import static java.time.format.DateTimeFormatter.ISO_OFFSET_DATE_TIME;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.apache.commons.lang3.StringUtils.trim;
import static org.apache.commons.lang3.StringUtils.trimToNull;

/**
 * Класс занимается преобразованием данных о сделках из формата транспорта в формат ядра
 */
@ParametersAreNonnullByDefault
class AdfoxDealConverter {

    @SuppressWarnings("WeakerAccess")
    static final int PREFERRED_DEAL = 10;
    @SuppressWarnings("WeakerAccess")
    static final int PRIVATE_MARKETPLACE = 11;

    Deal convertAdfoxDealCreate(AdfoxDealCreatePayload adfoxDeal) {
        String currencyId = adfoxDeal.getCurrencyId();
        long agencyRevenueRatio = adfoxDeal.getAgencyRevenueRatio();
        Deal result = new Deal();
        result.withDateCreated(LocalDateTime.now());
        result.withId(zeroToNull(adfoxDeal.getDealExportId()))
                .withClientId(zeroToNull(adfoxDeal.getAgencyId()))
                .withAdfoxStatus(convertStatus(adfoxDeal.getStatus()))
                .withDealType(convertType(adfoxDeal.getType()))
                // Валюта сделки
                .withCurrencyCode(convertCurrencyId(currencyId))
                // Заданные издателем параметры сделки
                .withAdfoxName(trimToNull(adfoxDeal.getName()))
                .withAdfoxDescription(trim(adfoxDeal.getDescription()))
                .withTargetingsText(trimToNull(adfoxDeal.getTargetingsText()))
                .withExpectedImpressionsPerWeek(zeroToNull(adfoxDeal.getExpectedImpressionsPerWeek()))
                .withExpectedMoneyPerWeek(zeroToNull(adfoxDeal.getExpectedMoneyPerWeek()))
                // Издатель и его контакты
                .withPublisherName(trimToNull(adfoxDeal.getPublisherName()))
                .withContacts(trimToNull(adfoxDeal.getContacts()))
                // Даты активности сделки
                .withDateStart(convertTimestamp(adfoxDeal.getDateTimeStartUtc()))
                .withDateEnd(convertTimestamp(adfoxDeal.getDateTimeEndUtc()))
                // Цены, коммиссии и вознаграждения
                .withCpm(convertCpmFromMicros(adfoxDeal.getCpm()))
                .withAgencyFeePercent(convertRatioFromMicros(agencyRevenueRatio))
                .withMarginRatio(convertRatioFromMicros(adfoxDeal.getMarginRatio()))
                // Дополнительные колонки
                .withPlacements(extractPlacements(adfoxDeal))
                .withDealJson(extractMisc(adfoxDeal));
        return result;
    }

    /**
     * Преобразует {@code 0}-значения в {@code null}.
     * Protobuf незаданные числовые параметры передаёт как {@code 0}.
     */
    @Nullable
    Long zeroToNull(Long protobufNumber) {
        return protobufNumber == 0 ? null : protobufNumber;
    }

    StatusAdfox convertStatus(AdfoxDealStatus status) {
        switch (status) {
            case created:
                return StatusAdfox.CREATED;
            case active:
                return StatusAdfox.ACTIVE;
            case closed:
                return StatusAdfox.CLOSED;
            default:
                throw new IllegalArgumentException("Unexpected value for Adfox deal status: " + status);
        }
    }

    DealType convertType(Integer type) {
        switch (type) {
            case PREFERRED_DEAL:
                return DealType.PREFERRED_DEAL;
            case PRIVATE_MARKETPLACE:
                return DealType.PRIVATE_MARKETPLACE;
            default:
                throw new IllegalArgumentException("Unexpected deal type: " + type);
        }
    }

    @Nullable
    CurrencyCode convertCurrencyId(String currencyId) {
        if (isEmpty(currencyId)) {
            return null;
        }
        return Currencies.getCurrency(Integer.parseInt(currencyId)).getCode();
    }

    LocalDateTime convertTimestamp(String dateTimeStartUtc) {
        if (StringUtils.isBlank(dateTimeStartUtc)) {
            return null;
        }
        return LocalDateTime.parse(dateTimeStartUtc, ISO_OFFSET_DATE_TIME);
    }

    /**
     * Преобразует микроединицы в единицы.
     * В транспорте все числа передаются домноженными на 1E6, чтобы избежать ошибок округления.
     */
    @Nullable
    BigDecimal convertNumberFromMicros(@Nullable Long micros) {
        if (micros == null || micros == 0) {
            return null;
        }
        return BigDecimal.valueOf(micros, 6);
    }

    /**
     * Преобразует микроединицы в единицы.
     * В транспорте все числа передаются домноженными на 1E6, чтобы избежать ошибок округления.
     */
    @Nullable
    Percent convertRatioFromMicros(Long agencyRevenueRatioProtoValue) {
        Long agencyRevenueRatio = zeroToNull(agencyRevenueRatioProtoValue);
        if (agencyRevenueRatio == null) {
            return null;
        }
        return Percent.fromRatio(convertNumberFromMicros(agencyRevenueRatio));
    }

    /**
     * Преобразует cpm из микроединиц
     * В транспорте все числа передаются домноженными на 1E6, чтобы избежать ошибок округления.
     * Для cpm не надо переводить значение в единицы, поскольку это единицы, домноженные на 1E3
     */
    @Nullable
    BigDecimal convertCpmFromMicros(@Nullable Long micros) {
        if (micros == null || micros == 0) {
            return null;
        }
        return BigDecimal.valueOf(micros, 3);
    }

    /**
     * Возвращает JSON-строку с дополнительными параметрами, которые не раскладываются в поля {@link Deal}
     */
    String extractMisc(AdfoxDealCreatePayload adfoxDeal) {
        return JsonUtils.toJson(CreateDealMiscFieldsConverter.extractMiscParameters(adfoxDeal));
    }

    /**
     * Возвращает JSON-строку с информацией о площадке {@link Deal}
     */
    List<DealPlacement> extractPlacements(AdfoxDealCreatePayload adfoxDeal) {
        return CreateDealPlacementsConverter.extractPlacements(adfoxDeal);
    }
}
