package ru.yandex.direct.jobs.adfox.messaging.handler;

import java.util.Collections;

import com.google.protobuf.Any;
import com.google.protobuf.InvalidProtocolBufferException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.adfoxmessaging.protos.AdfoxDealCreatePayload;
import ru.yandex.direct.adfoxmessaging.protos.CommonMessage;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.deal.model.Deal;
import ru.yandex.direct.core.entity.deal.service.DealService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.jobs.adfox.messaging.AdfoxMessageHandler;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.result.MassResult;

import static com.google.common.base.Preconditions.checkArgument;

@Component
public class DealCreateAdfoxMessageHandler implements AdfoxMessageHandler {
    private static final Logger logger = LoggerFactory.getLogger(DealCreateAdfoxMessageHandler.class);

    private final DealService dealService;
    private final ClientService clientService;
    private final AdfoxDealConverter adfoxDealConverter;

    @Autowired
    public DealCreateAdfoxMessageHandler(DealService dealService,
                                         ClientService clientService) {
        this.dealService = dealService;
        this.clientService = clientService;
        adfoxDealConverter = new AdfoxDealConverter();
    }

    @Override
    public String getType() {
        return "deal:create";
    }

    @Override
    public void handleMessage(CommonMessage message) throws InvalidProtocolBufferException {
        Any payload = message.getPayload();
        checkArgument(payload.is(AdfoxDealCreatePayload.class),
                "Given payload has unexpected type. Expected: %s. Got payload: %s",
                AdfoxDealCreatePayload.class, payload);

        AdfoxDealCreatePayload createPayload = payload.unpack(AdfoxDealCreatePayload.class);
        Deal newDeal = adfoxDealConverter.convertAdfoxDealCreate(createPayload);
        ClientId clientId = ClientId.fromLong(newDeal.getClientId());

        // Проверка валидности указанного clientId
        // Он должен указывать на Агенство
        Client client = clientService.getClient(clientId);
        checkArgument(client != null, "Can't find client by id %s", clientId);
        checkArgument(client.getRole() == RbacRole.AGENCY, "Given ClientID %s doesn't relates to Agency", clientId);

        logger.debug("Add deal with id {} to db", newDeal.getId());
        MassResult<Long> result = dealService.addDeals(clientId, Collections.singletonList(newDeal));
        //noinspection ConstantConditions
        checkArgument(result.isSuccessful() && !result.getValidationResult().hasAnyErrors(),
                "Can't create deal. Errors: %s", result.getValidationResult().flattenErrors());
    }

}
