package ru.yandex.direct.jobs.adfox.messaging.handler;

import com.google.protobuf.Any;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.adfoxmessaging.protos.AdfoxDealUpdatePayload;
import ru.yandex.direct.adfoxmessaging.protos.CommonMessage;
import ru.yandex.direct.core.entity.deal.model.DealAdfox;
import ru.yandex.direct.core.entity.deal.service.DealService;
import ru.yandex.direct.jobs.adfox.messaging.AdfoxMessageHandler;
import ru.yandex.direct.result.MassResult;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.singletonList;

@Component
public class DealUpdateAdfoxMessageHandler implements AdfoxMessageHandler {
    private static final Logger logger = LoggerFactory.getLogger(DealUpdateAdfoxMessageHandler.class);

    private final DealService dealService;
    private final AdfoxDealConverter adfoxDealConverter;

    @Autowired
    public DealUpdateAdfoxMessageHandler(DealService dealService) {
        this.dealService = dealService;
        adfoxDealConverter = new AdfoxDealConverter();
    }

    @Override
    public String getType() {
        return "deal:update";
    }

    @Override
    public void handleMessage(CommonMessage message) throws Exception {
        logger.debug("Processing {} message {}", getType(), message);

        Any payload = message.getPayload();
        checkArgument(payload.is(AdfoxDealUpdatePayload.class),
                "Given payload has unexpected type. Expected: %s. Got payload: %s",
                AdfoxDealUpdatePayload.class, payload);

        AdfoxDealUpdatePayload dealUpdatePayload = payload.unpack(AdfoxDealUpdatePayload.class);
        DealAdfox updateRequest = convertDealUpdateToModel(dealUpdatePayload);

        MassResult<Long> result = dealService.updateAdfoxDeals(singletonList(updateRequest));
        //noinspection ConstantConditions
        checkArgument(result.isSuccessful() && !result.getValidationResult().hasAnyErrors(),
                "Can't update deal. Errors: %s", result.getValidationResult().flattenErrors());
    }

    private DealAdfox convertDealUpdateToModel(AdfoxDealUpdatePayload payload) {
        return new DealAdfox()
                .withId(payload.getDealExportId())
                .withAdfoxStatus(adfoxDealConverter.convertStatus(payload.getStatus()));
    }
}
