package ru.yandex.direct.jobs.adfox.messaging.ytutils;

import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.yt.ytclient.proxy.ApiServiceClient;
import ru.yandex.yt.ytclient.rpc.RpcError;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

/**
 * Класс-помощник для чтения упорядоченных таблиц YT как очередей сообщений
 */
public class YtOrderedTableReader {
    private static final Logger logger = LoggerFactory.getLogger(YtOrderedTableReader.class);

    private final String tablePath;
    private final ApiServiceClient client;

    public YtOrderedTableReader(String tablePath, ApiServiceClient client) {
        this.tablePath = tablePath;
        this.client = client;
    }

    /**
     * Вычитывает {@code limit} строк со смещения {@code offset}, используя для преобразования строк
     * функцию, возвращённую {@code rowConverterSupplier}.
     * <p>
     * Для упорядоченных таблиц {@code offset} используется в условии {@code WHERE [$row_index] > offset}.
     */
    public <T> List<T> read(long offset, long limit,
                            Function<TableSchema, Function<UnversionedRow, T>> rowConverterSupplier) throws YtReadException {
        String query = String.format("* from [%s] where [$row_index] > %s limit %s", tablePath, offset, limit);
        logger.debug("Will query rows from {} by query: {}", tablePath, query);
        try {
            UnversionedRowset unversionedRowset = client.selectRows(query).get(15, TimeUnit.SECONDS);
            TableSchema schema = unversionedRowset.getSchema();
            Function<UnversionedRow, T> rowConverter = rowConverterSupplier.apply(schema);
            return unversionedRowset.getRows().stream().map(rowConverter).collect(Collectors.toList());
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        } catch (ExecutionException | TimeoutException | RpcError e) {
            throw new YtReadException("Can't get rows from " + tablePath, e);
        }
    }
}
