package ru.yandex.direct.jobs.adgeneration;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.JOBS_AD_GENERATION_KEYWORDS_LAST_START;

/**
 * Джоба раз в день мониторит и запускает yql-скрипт для обработки логов из ad_generation. Всего 6 скриптов
 * для логов: keywords, images, UC keywords, UC images, UC texts, UC sitelinks. Эта джоба обрабатывает keywords.
 * Джоба подхватывает подневные таблицы из //home/direct/logs/messages_log начиная со дня последнего запуска и ДО текущей даты.
 * Дата последнего запуска берется из ppc_properties.OBS_AD_GENERATION_KEYWORDS_LAST_START. Если записи нет,
 * то берется таблица за вчерашний день.
 * Результат джобы - 2 подневные таблицы на каждую подхваченную из messages_logs. Одна попадает в
 * //home/direct/logs/ad_generation/keywords/1d другая в //home/direct/logs/ad_generation/keywords/1d/statistic
 * Перед запуском нужно выставить дату в формате YYYY-MM-DD OBS_AD_GENERATION_KEYWORDS_LAST_START не позже, чем дата в имени
 * входной таблицы из //home/direct/logs/messages_log
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 48),
        tags = {CheckTag.DIRECT_PRIORITY_2, CheckTag.YT},
        notifications = {
                @OnChangeNotification(recipient = {NotificationRecipient.LOGIN_A_DUBOV, NotificationRecipient.LOGIN_AMMSAID},
                        status = {JugglerStatus.OK, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM)
        },
        needCheck = ProductionOnly.class
)
@Hourglass(cronExpression = "0 42 4 * * ?", needSchedule = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class AdGenerationKeywordsLogsJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(AdGenerationKeywordsLogsJob.class);

    private static final YtCluster YT_CLUSTER = YtCluster.HAHN;
    private static final String LOG_TYPE = "keywords";
    private static final String QUERY_PATH = "adgeneration/keywords.sql";
    private static final String DST_FOLDER_REL_PATH = "keywords/1d";

    private final AdGenerationLogsService adGenerationLogsService;

    @Autowired
    public AdGenerationKeywordsLogsJob(AdGenerationLogsService adGenerationLogsService) {
        this.adGenerationLogsService = adGenerationLogsService;
    }

    @Override
    public void execute() {
        adGenerationLogsService.processAdGenerationLogs(YT_CLUSTER, LOG_TYPE, JOBS_AD_GENERATION_KEYWORDS_LAST_START,
                QUERY_PATH, DST_FOLDER_REL_PATH);
        logger.info("AdGeneration logs for {} processed", LOG_TYPE);
    }
}
