package ru.yandex.direct.jobs.adgeneration;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YqlQuery;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;

@Component
@ParametersAreNonnullByDefault
public class AdGenerationLogsService {
    private static final Logger logger = LoggerFactory.getLogger(AdGenerationLogsService.class);

    private final YtProvider ytProvider;
    private final DirectConfig adGenerationLogsConfig;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    public AdGenerationLogsService(YtProvider ytProvider, DirectConfig directConfig,
                                   PpcPropertiesSupport ppcPropertiesSupport) {
        this.ytProvider = ytProvider;
        this.adGenerationLogsConfig = directConfig.getBranch("ad_generation_logs");
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    public void processAdGenerationLogs(YtCluster ytCluster, String logType, PpcPropertyEnum ppcProperty,
                                        String queryPath, String dstFolderRelPath) {
        var ytOperator = ytProvider.getOperator(ytCluster);
        var messageLogsPath = adGenerationLogsConfig.getString("message_logs_path");
        var lastStartDate = readLastStartOrYesterday(ppcProperty);

        var currentDate = LocalDate.now();

        var tables = getTablesList(ytOperator, messageLogsPath, lastStartDate, currentDate);
        if (tables.isEmpty()) {
            logger.info("No new data found from message logs");
            return;
        }

        String query = String.join("\n", new ClassPathResourceInputStreamSource(queryPath).readLines());
        var dstPath = adGenerationLogsConfig.getString("dst_path");
        var dstFolder = YtPathUtil.generatePath(dstPath, dstFolderRelPath);

        for (var table : tables) {
            try (var ignore = Trace.current().profile("ad_generation:yql", logType)) {
                ytOperator.yqlExecute(new YqlQuery(query, messageLogsPath, table, dstFolder)
                        .withTitle(String.format("ad_generation:%s", logType)));
            }
        }

        writeLastStart(ppcProperty, currentDate);
    }

    private List<String> getTablesList(YtOperator ytOperator, String folder, LocalDate dateFrom, LocalDate dateTo) {
        var tables = new ArrayList<String>();
        var cypress = ytOperator.getYt().cypress();

        var folderPath = YPath.simple(folder);
        if (!cypress.exists(folderPath)) {
            logger.warn("Yt folder {} not found", folder);
            return tables;
        }

        var nodes = cypress.list(folderPath);
        logger.info("Found nodes {} in folder {}", nodes, folder);

        for (var node : nodes) {
            LocalDate tableDate;
            try {
                tableDate = LocalDate.parse(node.stringValue(), DateTimeFormatter.ISO_LOCAL_DATE);
            } catch (DateTimeParseException ex) {
                logger.warn("Cannot parse datetime from {}. Exception: {}", node.stringValue(), ex);
                continue;
            }

            if ((tableDate.isEqual(dateFrom) || tableDate.isAfter(dateFrom)) && tableDate.isBefore(dateTo)) {
                tables.add(node.stringValue());
            }
        }

        logger.info("Collected tables for processing {}", tables);
        return tables;
    }

    private LocalDate readLastStartOrYesterday(PpcPropertyEnum ppcProperty) {
        var lastStartString = ppcPropertiesSupport.get(ppcProperty.getName());
        LocalDate lastStart = LocalDate.now().minusDays(1);
        if (lastStartString == null) {
            return lastStart;
        }

        try {
            lastStart = LocalDate.parse(lastStartString, DateTimeFormatter.ISO_LOCAL_DATE);
        } catch (DateTimeParseException ex) {
            logger.warn("Cannot parse date {} from ppc_properties", lastStartString);
        }

        return lastStart;
    }

    private void writeLastStart(PpcPropertyEnum ppcProperty, LocalDate date) {
        ppcPropertiesSupport.set(ppcProperty.getName(), date.format(DateTimeFormatter.ISO_LOCAL_DATE));
    }
}
