package ru.yandex.direct.jobs.adgeneration;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.JOBS_AD_GENERATION_UC_IMAGES_LAST_START;

/**
 * Джоба раз в день мониторит и запускает yql-скрипт для обработки логов из ad_generation. Всего 6 скриптов
 * для логов: keywords, images, UC keywords, UC images, UC texts, UC sitelinks. Эта джоба обрабатывает UC images.
 * Джоба подхватывает подневные таблицы из //home/direct/logs/messages_log начиная со дня последнего запуска и ДО текущей даты.
 * Дата последнего запуска берется из ppc_properties.JOBS_AD_GENERATION_UC_IMAGES_LAST_START. Если записи нет,
 * то берется таблица за вчерашний день.
 * Результат джобы - подневная таблица на каждую подхваченную из messages_logs, которая попадает в
 * //home/direct/logs/ad_generation/UC/images/byDays и дозапись строки в текущую
 * //home/direct/logs/ad_generation/UC/images/summary
 * Перед запуском нужно:
 * 1. дата в формате YYYY-MM-DD JOBS_AD_GENERATION_UC_IMAGES_LAST_START не позже, чем дата в имени входной таблицы
 * из //home/direct/logs/messages_log.
 * 2. таблица //home/direct/logs/ad_generation/UC/images/summary в которой копятся данные. Строки каждой даты
 * записываются в нее только один раз и не перезаписываются.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 48),
        tags = {CheckTag.DIRECT_PRIORITY_2, CheckTag.YT},
        notifications = {
                @OnChangeNotification(recipient = {NotificationRecipient.LOGIN_A_DUBOV, NotificationRecipient.LOGIN_AMMSAID},
                        status = {JugglerStatus.OK, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM)
        },
        needCheck = ProductionOnly.class
)
@Hourglass(cronExpression = "0 44 4 * * ?", needSchedule = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class AdGenerationUCImagesLogsJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(AdGenerationUCImagesLogsJob.class);

    private static final YtCluster YT_CLUSTER = YtCluster.HAHN;
    private static final String LOG_TYPE = "uc_images";
    private static final String QUERY_PATH = "adgeneration/ucimages.sql";
    private static final String DST_FOLDER_REL_PATH = "UC/images";

    private final AdGenerationLogsService adGenerationLogsService;

    @Autowired
    public AdGenerationUCImagesLogsJob(AdGenerationLogsService adGenerationLogsService) {
        this.adGenerationLogsService = adGenerationLogsService;
    }

    @Override
    public void execute() {
        adGenerationLogsService.processAdGenerationLogs(YT_CLUSTER, LOG_TYPE, JOBS_AD_GENERATION_UC_IMAGES_LAST_START,
                QUERY_PATH, DST_FOLDER_REL_PATH);
        logger.info("AdGeneration logs for {} processed", LOG_TYPE);
    }
}
