package ru.yandex.direct.jobs.advq.offline.dataimport;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupShowsForecast;
import ru.yandex.direct.core.entity.keyword.model.KeywordForecast;
import ru.yandex.direct.jobs.advq.offline.processing.OfflineAdvqProcessingBaseTableRow;

/**
 * Консьюмер, обрабатывающий строки с прогнозом показов ключевых фраз из YT.
 * Собирает данные в пачки по группам и, после окончания работы, передает описание прогноза показов во внутреннем формате
 * следующему потребителю
 * <p>
 * Предполагается, что один консьюмер получит всю информацию о прогнозе в отдельной группе
 * (то есть группа не может попасть в два разных консьюмера) и поэтому отдаваемые им прогнозы во внутрннем формате могут
 * быть записаны в базу данных без дополнительной аггрегации
 */
@ParametersAreNonnullByDefault
class OfflineAdvqImportRowConsumer implements Consumer<OfflineAdvqProcessingBaseTableRow> {
    private static final Logger logger = LoggerFactory.getLogger(OfflineAdvqImportRowConsumer.class);

    private final long totalRowCount;
    private final long startTime;

    private Map<Long, List<KeywordForecast>> groupIdToData;
    private Map<Long, String> groupIdToGeo;
    private long rowsProcessed;

    OfflineAdvqImportRowConsumer() {
        this(0);
    }

    OfflineAdvqImportRowConsumer(long totalRowCount) {
        this.totalRowCount = totalRowCount;
        this.startTime = getTimestampSeconds();

        groupIdToData = new HashMap<>();
        groupIdToGeo = new HashMap<>();
        rowsProcessed = 0;
    }

    private static long getTimestampSeconds() {
        return System.currentTimeMillis() / 1000L;
    }

    private long getSecondsLeft(long timestamp) {
        if (rowsProcessed == 0) {
            return 0;
        }
        double timeWorking = timestamp - (double) startTime;
        return Math.round(timeWorking / rowsProcessed * totalRowCount);
    }

    Map<Long, List<KeywordForecast>> getGroupIdToData() {
        return groupIdToData;
    }

    Map<Long, String> getGroupIdToGeo() {
        return groupIdToGeo;
    }

    /**
     * Подготовить собранные данные и отправить их следующему потребителю
     */
    List<AdGroupShowsForecast> getDataAndCleanup() {
        logger.info("Returning data for {} groups", groupIdToData.size());
        long timestamp = getTimestampSeconds();
        logger.info(
                "Processed {} rows of {}, importing for {} seconds, approx {} seconds left",
                rowsProcessed, totalRowCount, Duration.ofSeconds(timestamp - startTime),
                Duration.ofSeconds(getSecondsLeft(timestamp)));

        if (groupIdToData.isEmpty()) {
            return Collections.emptyList();
        }

        List<AdGroupShowsForecast> forecasts = groupIdToData.entrySet().stream()
                .map(e -> new AdGroupShowsForecast()
                        .withId(e.getKey())
                        .withGeo(groupIdToGeo.get(e.getKey()))
                        .withKeywordForecasts(e.getValue()))
                .collect(Collectors.toList());

        groupIdToData = new HashMap<>();
        groupIdToGeo = new HashMap<>();

        return forecasts;
    }

    @Override
    public void accept(OfflineAdvqProcessingBaseTableRow tableRow) {
        rowsProcessed++;
        long rowGroupId = tableRow.getGroupId();

        List<KeywordForecast> keywordForecasts = groupIdToData.computeIfAbsent(rowGroupId, i -> new ArrayList<>());
        if (tableRow.getId() != 0L) {
            keywordForecasts.add(
                    new KeywordForecast()
                            .withId(tableRow.getId())
                            .withKeyword(tableRow.getOriginalKeyword())
                            .withShowsForecast(tableRow.getForecast()));
        }

        String geo = tableRow.getGeo();
        if (geo != null && !"".equals(geo)) {
            groupIdToGeo.putIfAbsent(rowGroupId, geo);
        }
    }
}
