package ru.yandex.direct.jobs.advq.offline.dataimport;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.util.RelaxedWorker;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupShowsForecast;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupsShowsForecastService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.advq.offline.processing.OfflineAdvqProcessingBaseTableRow;
import ru.yandex.direct.jobs.advq.offline.processing.OfflineAdvqProcessingJob;
import ru.yandex.direct.jobs.advq.offline.processing.OfflineAdvqProcessingMRSpec;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;

import static ru.yandex.direct.jobs.advq.offline.processing.OfflineAdvqProcessingJob.FORECAST_TIME_PROP_NAME;
import static ru.yandex.direct.jobs.advq.offline.processing.OfflineAdvqProcessingJob.UPLOAD_TIME_PROP_NAME;
import static ru.yandex.direct.jobs.configuration.JobsEssentialConfiguration.DEFAULT_YT_CLUSTER;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;

/**
 * Задача, загружающая из YT в базу данных Директа данные о прогнозе показов, посчитанные в offline ADVQ
 *
 * @see AdGroupsShowsForecastService#updateAdGroupsShowsForecast(int, List, LocalDateTime)
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 1),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_1, CheckTag.YT, CheckTag.GROUP_INTERNAL_SYSTEMS}
)
@Hourglass(periodInSeconds = 2400, needSchedule = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class OfflineAdvqKeywordsImportJob extends DirectShardedJob {
    static final String IMPORT_TIME_PROP_NAME =
            String.format("%s_last_import_time", OfflineAdvqProcessingJob.class.getSimpleName());

    private static final int YT_CHUNKS_MAX_SIZE = 1000000;
    private static final double TIME_WASTE_COEF = 2.0;
    private static final int UPDATE_CHUNK_SIZE = 1000;

    private static final Logger logger = LoggerFactory.getLogger(OfflineAdvqKeywordsImportJob.class);

    private final ShardHelper shardHelper;
    private final YtProvider ytProvider;
    private final YtCluster defaultYtCluster;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final AdGroupsShowsForecastService adGroupsShowsForecastService;

    @Autowired
    public OfflineAdvqKeywordsImportJob(ShardHelper shardHelper, YtProvider ytProvider,
                                        @Qualifier(DEFAULT_YT_CLUSTER) YtCluster defaultYtCluster,
                                        PpcPropertiesSupport ppcPropertiesSupport,
                                        AdGroupsShowsForecastService adGroupsShowsForecastService) {
        this.shardHelper = shardHelper;
        this.ytProvider = ytProvider;
        this.defaultYtCluster = defaultYtCluster;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.adGroupsShowsForecastService = adGroupsShowsForecastService;
    }

    OfflineAdvqKeywordsImportJob(int shard, ShardHelper shardHelper, YtProvider ytProvider,
                                 YtCluster defaultYtCluster, PpcPropertiesSupport ppcPropertiesSupport,
                                 AdGroupsShowsForecastService adGroupsShowsForecastService) {
        super(shard);
        this.shardHelper = shardHelper;
        this.ytProvider = ytProvider;
        this.defaultYtCluster = defaultYtCluster;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.adGroupsShowsForecastService = adGroupsShowsForecastService;
    }

    @Override
    public void execute() {
        YtClusterConfig ytClusterConfig = ytProvider.getClusterConfig(defaultYtCluster);
        YtTable table =
                new YtTable(OfflineAdvqProcessingMRSpec.getShardTableName(ytClusterConfig.getHome(), getShard()));

        LocalDateTime forecastCalcDateTime = getTableForecastDateTime();
        String lastForecastUploadTime = fetchCurrentUploadTime();
        if (lastForecastUploadTime == null || forecastCalcDateTime == null) {
            logger.info("No ready data");
            return;
        }
        if (!forecastNeedsUpdate(lastForecastUploadTime)) {
            logger.info("{} table is already synced", table);
            return;
        }

        logger.info("Working in table {} with forecast of {}", table.getPath(), forecastCalcDateTime);
        importYtTableDataToDirectDataBase(table, forecastCalcDateTime);

        setLastUploadedProp(lastForecastUploadTime);
    }

    /**
     * Получить время загрузки таблицы из YT
     */
    @Nullable
    private String fetchCurrentUploadTime() {
        String currentUploadTime = ppcPropertiesSupport.get(UPLOAD_TIME_PROP_NAME);
        logger.info("Shard {} table upload time is {}", getShard(), currentUploadTime);
        return currentUploadTime;
    }

    /**
     * Проверить, что данные из таблицы с прогнозом показов еще не загружались в базу Директа
     */
    boolean forecastNeedsUpdate(String lastForecastUploadTime) {
        String lastImportTime = ppcPropertiesSupport.get(getLastImportTimePropName());
        logger.info("Last table sync was when upload time was {}", lastImportTime);
        return !lastForecastUploadTime.equals(lastImportTime);
    }

    /**
     * Записать дату последней загрузки данных в Базу
     */
    void setLastUploadedProp(String lastForecastUploadTime) {
        String propName = getLastImportTimePropName();
        logger.info("Setting prop {} to {}", propName, lastForecastUploadTime);
        ppcPropertiesSupport.set(propName, lastForecastUploadTime);
    }

    private LocalDateTime getTableForecastDateTime() {
        String s = ppcPropertiesSupport.get(FORECAST_TIME_PROP_NAME);
        logger.info("Shard {} table forecast time is {}", getShard(), s);
        return s == null ? null : LocalDateTime.parse(s);
    }

    /**
     * Прочитать все данные из таблицы с прогнозом показов и загрузить их в базу Директа
     */
    private void importYtTableDataToDirectDataBase(YtTable table, LocalDateTime forecastDateTime) {
        List<OffsetPair> selectChunks = prepareSelectChunks();
        int size = selectChunks.size();
        logger.info("Totally {} chunks in queue", size);

        YtOperator ytOperator = ytProvider.getOperator(defaultYtCluster);
        RelaxedWorker relaxedWorker = new RelaxedWorker(TIME_WASTE_COEF);
        OfflineAdvqImportRowConsumer rowConsumer =
                new OfflineAdvqImportRowConsumer(ytOperator.readTableNumericAttribute(table, "row_count"));
        int taskNum = 0;
        for (OffsetPair selectChunk : selectChunks) {
            logger.info("Selecting chunk with groups with id from {} to {} ({}/{})", selectChunk.getOffsetStart(),
                    selectChunk.getOffsetEnd(), ++taskNum, size);

            List<AdGroupShowsForecast> forecasts = getForecastsFromYt(ytOperator, table, rowConsumer, selectChunk);
            for (List<AdGroupShowsForecast> chunk : Lists.partition(forecasts, UPDATE_CHUNK_SIZE)) {
                relaxedWorker.runAndRelax(() -> adGroupsShowsForecastService
                        .updateAdGroupsShowsForecast(getShard(), chunk, forecastDateTime));
            }
        }
    }

    private List<AdGroupShowsForecast> getForecastsFromYt(YtOperator ytOperator, YtTable table,
                                                          OfflineAdvqImportRowConsumer rowConsumer, OffsetPair selectChunk) {
        ytOperator
                .readTableByKeyRange(table, rowConsumer, new OfflineAdvqProcessingBaseTableRow(),
                        selectChunk.getOffsetStart(),
                        selectChunk.getOffsetEnd());
        return rowConsumer.getDataAndCleanup();
    }

    private List<OffsetPair> prepareSelectChunks() {
        Long maxGroupId = shardHelper.getMaxAdGroupId();
        logger.info("Max groupId is {}", maxGroupId);
        logger.info("Processing {} groups in chunk", YT_CHUNKS_MAX_SIZE);

        List<OffsetPair> selectChunks = new ArrayList<>();
        long currentOffset = 0;
        while (currentOffset <= maxGroupId) {
            selectChunks.add(new OffsetPair(currentOffset, currentOffset + YT_CHUNKS_MAX_SIZE));

            currentOffset = currentOffset + YT_CHUNKS_MAX_SIZE;
        }

        // Перемешиваем, чтобы нагрузка была равномернее
        Collections.shuffle(selectChunks);
        return selectChunks;
    }

    private String getLastImportTimePropName() {
        return IMPORT_TIME_PROP_NAME + "_s" + getShard();
    }
}
