package ru.yandex.direct.jobs.advq.offline.export;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.direct.ytwrapper.specs.OperationSpec;

import static ru.yandex.direct.jobs.configuration.JobsEssentialConfiguration.DEFAULT_YT_CLUSTER;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;

/**
 * Задача, выгружающая все фразы неархивных кампаний для подсчета прогноза показов в офлайновом ADVQ.
 * В процессе выгрузки из ключевой фразы убираются минус-фразы, которые впоследствии объединяются с
 * минус-фразами на группу и из получившего списка удаляются все пересечения с основной ключевой фразой.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2, hours = 12),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_1, CheckTag.YT, CheckTag.GROUP_INTERNAL_SYSTEMS}
)
@Hourglass(cronExpression = "38 10 22 * * ?", needSchedule = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class OfflineAdvqKeywordsExportJob extends DirectJob {
    private static final String LAST_UPLOADED_PROP_TEMPLATE = "%s_%s_last_upload_time";

    /**
     * Служебная структура содержащая информацию об актуальной дате и времени заливки таблицы и
     * дате и времени заливки таблицы на момент проведения последнего экспорта фраз.
     */
    static class TableUploadTimes {
        final String currentUploadTime;
        final String uploadTimeOnLastUpdate;

        TableUploadTimes(String currentUploadTime, @Nullable String uploadTimeOnLastUpdate) {
            this.currentUploadTime = currentUploadTime;
            this.uploadTimeOnLastUpdate = uploadTimeOnLastUpdate;
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(OfflineAdvqKeywordsExportJob.class);

    private final YtProvider ytProvider;
    private final YtCluster defaultYtCluster;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final StopWordService stopWordService;

    private YtOperator ytOperator;
    private YtClusterConfig ytClusterConfig;

    @Autowired
    public OfflineAdvqKeywordsExportJob(YtProvider ytProvider,
                                        @Qualifier(DEFAULT_YT_CLUSTER) YtCluster defaultYtCluster,
                                        PpcPropertiesSupport ppcPropertiesSupport,
                                        StopWordService stopWordService) {
        this.ytProvider = ytProvider;
        this.defaultYtCluster = defaultYtCluster;

        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.stopWordService = stopWordService;
    }

    @Override
    public void execute() {
        ytClusterConfig = ytProvider.getClusterConfig(defaultYtCluster);
        ytOperator = ytProvider.getOperator(defaultYtCluster);

        Map<YtTable, TableUploadTimes> tableToUploadTimes = getUploadTimes(ytOperator, OfflineAdvqMRSpec.getMRTables());

        logger.info("Checking if export table needs update");
        if (exportTableNeedsUpdate(tableToUploadTimes)) {
            runMapReduce();
            setNewUploadTimesProperties(tableToUploadTimes);
        }
    }

    /**
     * Запустить задачу в MR-кластере
     */
    private void runMapReduce() {
        logger.info("Preparing map-reduce job");
        OperationSpec spec = OfflineAdvqMRSpec.getSpec(ytClusterConfig.getHome(), stopWordService.getStopWords());
        logger.info("Starting map-reduce job");
        ytOperator.runOperation(spec);
        logger.info("Map-reduce finished successfully");
    }

    /**
     * Проверить, требуется ли проведение экспорта фраз для офлайнового ADVQ. Возвращает истинное значение в случае,
     * когда хотя бы одна из используемых для экспорта таблиц изменилась
     *
     * @param tableToUploadTimes маппинг YT-таблицы в структуру с таимстампами
     */
    boolean exportTableNeedsUpdate(Map<YtTable, TableUploadTimes> tableToUploadTimes) {
        for (Map.Entry<YtTable, TableUploadTimes> entry : tableToUploadTimes.entrySet()) {
            TableUploadTimes times = entry.getValue();
            if (times.uploadTimeOnLastUpdate == null) {
                logger.info("Looks like it's the first time the job runs. Need to generate a table for offline ADVQ");
                return true;
            } else if (!times.uploadTimeOnLastUpdate.equals(times.currentUploadTime)) {
                logger.info(
                        "{} table was updated after the previous generation of table for offline ADVQ. Need to regenerate it",
                        entry.getKey().ypath());
                return true;
            }
        }
        logger.info("There is no need to regenerate table for offline ADVQ");
        return false;
    }

    /**
     * Получить маппинг YT-таблицы в структуру {@link TableUploadTimes}.
     * Даныне берутся из YT и из старых данных, записанных в property
     */
    Map<YtTable, TableUploadTimes> getUploadTimes(YtOperator ytOperator, List<YtTable> tables) {
        Map<YtTable, TableUploadTimes> result = new HashMap<>();
        for (YtTable ytTable : tables) {
            String uploadTimeOnLastUpdate = ppcPropertiesSupport.get(
                    String.format(LAST_UPLOADED_PROP_TEMPLATE, this.getClass().getSimpleName(), ytTable.getName()));
            String currentUploadTime = ytOperator.readTableUploadTime(ytTable);
            if (logger.isInfoEnabled()) {
                logger.info("{} table current upload time is {}; last sync was at {}", ytTable.getPath(),
                        currentUploadTime, uploadTimeOnLastUpdate);
            }
            result.put(ytTable, new TableUploadTimes(currentUploadTime, uploadTimeOnLastUpdate));
        }
        return result;
    }

    /**
     * Установить новые значения для property с таимстампами загрузки таблиц, использованных для эспорта
     *
     * @param tableToUploadTimes маппинг YT-таблицы в структуру с таимстампами
     */
    void setNewUploadTimesProperties(Map<YtTable, TableUploadTimes> tableToUploadTimes) {
        for (Map.Entry<YtTable, TableUploadTimes> entry : tableToUploadTimes.entrySet()) {
            ppcPropertiesSupport.set(
                    String.format(LAST_UPLOADED_PROP_TEMPLATE, this.getClass().getSimpleName(),
                            entry.getKey().getName()),
                    entry.getValue().currentUploadTime);
        }
    }
}
