package ru.yandex.direct.jobs.advq.offline.export;

import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.direct.ytwrapper.model.YtTableIndexMapper;
import ru.yandex.direct.ytwrapper.model.YtTableRow;
import ru.yandex.direct.ytwrapper.model.attributes.CompressionCodecAttr;
import ru.yandex.direct.ytwrapper.model.attributes.OptimizeForAttr;
import ru.yandex.direct.ytwrapper.specs.AppendableSpecBuilder;
import ru.yandex.direct.ytwrapper.specs.MapReduceSpecBuilder;
import ru.yandex.direct.ytwrapper.specs.OperationSpec;
import ru.yandex.direct.ytwrapper.specs.SetTableAttributesSpecBuilder;
import ru.yandex.direct.ytwrapper.specs.SortSpecBuilder;
import ru.yandex.direct.ytwrapper.tables.generated.YtBids;
import ru.yandex.direct.ytwrapper.tables.generated.YtBidsRow;
import ru.yandex.direct.ytwrapper.tables.generated.YtCampaigns;
import ru.yandex.direct.ytwrapper.tables.generated.YtCampaignsRow;
import ru.yandex.direct.ytwrapper.tables.generated.YtDbTables;
import ru.yandex.direct.ytwrapper.tables.generated.YtPhrases;
import ru.yandex.direct.ytwrapper.tables.generated.YtPhrasesRow;
import ru.yandex.misc.dataSize.DataSize;

import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.ytwrapper.specs.AppendableSpecBuilder.pair;

/**
 * Общий код для маппера и редьюсера
 */
@ParametersAreNonnullByDefault
class OfflineAdvqMRSpec {
    /**
     * Относительный путь к выгружаемой таблице
     */
    private static final Duration MR_TIMEOUT = Duration.ofHours(12);
    private static final String OUTPUT_PATH = "export/phrases_for_forecast";

    /**
     * Список таблиц, учавствующих в экспорте специально вынесен отдельно и сделан статическим так как это позволяет
     * - получить объект таблицы с заполненными данными по индексу таблицы
     * - избежать необходимости сериализации маппера и редьюсера
     */
    private static final List<AppendableSpecBuilder.TableRowPair> mrTables = Collections.unmodifiableList(Arrays.asList(
            pair(YtDbTables.CAMPAIGNS, new YtCampaignsRow(Arrays.asList(
                    YtCampaigns.CID,
                    YtCampaigns.ARCHIVED))),
            pair(YtDbTables.PHRASES, new YtPhrasesRow(Arrays.asList(
                    YtPhrases.CID,
                    YtPhrases.PID,
                    YtPhrases.MW_TEXT,
                    YtPhrases.ADGROUP_TYPE,
                    YtPhrases.STATUSSHOWSFORECAST,
                    YtPhrases.FORECASTDATE,
                    YtPhrases.GEO))),
            pair(YtDbTables.BIDS, new YtBidsRow(Arrays.asList(
                    YtBids.CID,
                    YtBids.PID,
                    YtBids.ID,
                    YtBids.PHRASE)))
    ));

    private OfflineAdvqMRSpec() {
    }

    /**
     * Получить список таблиц из которых получается информация для экспорта фраз
     */
    static List<YtTable> getMRTables() {
        return mapList(mrTables, AppendableSpecBuilder.TableRowPair::getTable);
    }

    /**
     * Получить список описаний рядов таблиц из которых получается информация для экспорта фраз
     */
    static List<YtTableRow> getMRTablesRows() {
        return mapList(mrTables, AppendableSpecBuilder.TableRowPair::getRow);
    }

    /**
     * Создать и заполнить билдер спецификации MR-задачи
     * <p>
     * Здесь специально не устанавливаются таблица вывода для MR, и таблица ввода для сортировки, так как, в случае их
     * отсутствия, временная таблица для передачи будет добавлена и удалена после выполнения автоматически.
     */
    static OperationSpec getSpec(String outputHome, Set<String> stopWords) {
        MapReduceSpecBuilder mapReduceSpecBuilder = new MapReduceSpecBuilder()
                .addInputTables(mrTables)
                .setMapper(new YtTableIndexMapper())
                .addReduceByField(YtCampaigns.CID)
                .addSortByField(YtPhrases.CID)
                .addSortByField(YtPhrases.PID)
                .addSortByField(YtTableRow.TI_FIELD)
                .setReducer(new OfflineAdvqReducer(stopWords))
                .setReducerMemoryLimit(DataSize.fromMegaBytes(1500))
                .setMaxDataSizePerJob(DataSize.fromMegaBytes(100))
                .setPartitionCount(8192)
                .setPartitionJobCount(8192);

        new SortSpecBuilder()
                .addSortByField(OfflineAdvqExportOutputTableRow.ID)
                .appendTo(mapReduceSpecBuilder);

        new SetTableAttributesSpecBuilder()
                .setOutputTable(new YtTable(YtPathUtil.generatePath(outputHome, relativePart(), OUTPUT_PATH)))
                .setOptimizeFor(OptimizeForAttr.SCAN)
                .setCompressionCodec(CompressionCodecAttr.BROTLI_8)
                .setSchema(new OfflineAdvqExportOutputTableRow())
                .appendTo(mapReduceSpecBuilder);

        return mapReduceSpecBuilder
                .setOperationTimeout(MR_TIMEOUT)
                .build();
    }
}
