package ru.yandex.direct.jobs.advq.offline.export;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.util.concurrent.UncheckedExecutionException;

import ru.yandex.advq.query.IllegalQueryException;
import ru.yandex.direct.core.entity.adgroup.model.StatusShowsForecast;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived;
import ru.yandex.direct.libs.keywordutils.inclusion.KeywordInclusionUtils;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordWithLemmasFactory;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.ytwrapper.model.YtReducer;
import ru.yandex.direct.ytwrapper.model.YtTableRow;
import ru.yandex.direct.ytwrapper.model.YtYield;
import ru.yandex.direct.ytwrapper.tables.generated.YtBidsRow;
import ru.yandex.direct.ytwrapper.tables.generated.YtCampaigns;
import ru.yandex.direct.ytwrapper.tables.generated.YtCampaignsRow;
import ru.yandex.direct.ytwrapper.tables.generated.YtPhrasesRow;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeIgnoreField;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeObject;
import ru.yandex.inside.yt.kosher.operations.Statistics;
import ru.yandex.inside.yt.kosher.operations.Yield;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;

/**
 * Редьюсер для экспорта фраз для офлайн ADVQ. Полагается на то, что данные отсортированы по ID кампаний и групп,
 * а внутри группы сначала идет информация о самой группе, а только затем о фразах.
 * Отдает данные в формате {@link OfflineAdvqExportOutputTableRow}.
 */
@YTreeObject
@ParametersAreNonnullByDefault
public class OfflineAdvqReducer extends YtReducer<Long> {
    private static final String MOBILE_CONTENT_GROUP = "mobile_content";
    private static final String DEVICE_PHONE = "phone";
    private static final String DEVICE_TABLET = "tablet";
    private static final Pattern SPLIT_PATTERN = Pattern.compile("\\s+-");

    private final Set<String> stopWords;

    @YTreeIgnoreField
    private KeywordWithLemmasFactory keywordFactory = new KeywordWithLemmasFactory(100000, 10000, 10000);
    @YTreeIgnoreField
    private List<YtTableRow> rowsOrder = OfflineAdvqMRSpec.getMRTablesRows();

    @YTreeIgnoreField
    private Long currentCid = null;

    @YTreeIgnoreField
    private Long currentGroupId = null;
    @YTreeIgnoreField
    private String currentGeo = null;
    @YTreeIgnoreField
    private List<String> groupMinusPhrases = null;
    @YTreeIgnoreField
    private List<String> devices = null;

    public OfflineAdvqReducer(Set<String> stopWords) {
        super();
        this.stopWords = stopWords;
    }

    public OfflineAdvqReducer(YtYield ytYield, Set<String> stopWords) {
        super(ytYield);
        this.stopWords = stopWords;
    }

    @Override
    public void start(Yield<YTreeMapNode> yield, Statistics statistics) {
        super.start(yield, statistics);
        if (rowsOrder == null) {
            rowsOrder = OfflineAdvqMRSpec.getMRTablesRows();
        }
        if (keywordFactory == null) {
            keywordFactory = new KeywordWithLemmasFactory(100000, 10000, 10000);
        }
    }

    @Override
    public Long key(YtTableRow row) {
        return row.valueOf(YtCampaigns.CID);
    }

    @Override
    public void reduce(Long cid, Iterator<YtTableRow> entries) {
        while (entries.hasNext()) {
            YtTableRow row = entries.next();
            int tableIndex = row.getTableIndex();
            YtTableRow realRow = rowsOrder.get(tableIndex);
            realRow.setDataFrom(row);

            if (realRow instanceof YtCampaignsRow) {
                String archived = ((YtCampaignsRow) realRow).getArchived();
                if (archived == null || CampaignsArchived.Yes.getLiteral().equalsIgnoreCase(archived)) {
                    break;
                }

                processCampaign((YtCampaignsRow) realRow);
            } else if (realRow instanceof YtPhrasesRow) {
                String statusShowsForecast = ((YtPhrasesRow) realRow).getStatusShowsForecast();
                if (currentCid == null || statusShowsForecast == null ||
                        StatusShowsForecast.toSource(StatusShowsForecast.ARCHIVED).getLiteral()
                                .equalsIgnoreCase(statusShowsForecast)) {
                    break;
                }
                processPhrase((YtPhrasesRow) realRow);
            } else if (realRow instanceof YtBidsRow) {
                YtBidsRow bid = (YtBidsRow) realRow;
                if (currentGroupId == null || !currentGroupId.equals(bid.getPid())) {
                    currentGroupId = null;
                    groupMinusPhrases = null;
                } else {
                    processBid(bid);
                }
            } else {
                throw new RuntimeException("Unexpected table");
            }
        }
        currentCid = null;
        currentGroupId = null;
        currentGeo = null;
        groupMinusPhrases = null;
        devices = null;
    }

    /**
     * Обработать строку из таблицы с кампаниями
     */
    private void processCampaign(YtCampaignsRow campaign) {
        currentCid = campaign.getCid();
    }

    private List<String> cleanMinusWords(Collection<String> minusWords) {
        return minusWords.stream()
                .map(String::trim)
                .filter(s -> !"".equals(s))
                .collect(Collectors.toList());
    }

    /**
     * Распарсить JSON с минус-фразами
     */
    private List<String> getParsedMinusWords(@Nullable String minusWords) {
        if (minusWords == null || Objects.equals(minusWords, "") || Objects.equals(minusWords, "[]")) {
            return Collections.emptyList();
        } else {
            return cleanMinusWords(Arrays.asList(JsonUtils.fromJson(minusWords, String[].class)));
        }
    }

    /**
     * Обработать строку из таблицы с группами. Предполагается, что вся информация о кампании у нас уже есть.
     */
    private void processPhrase(YtPhrasesRow group) {
        currentGroupId = group.getPid();
        groupMinusPhrases = getParsedMinusWords(group.getMwText());
        currentGeo = group.getGeo();
        if (currentGeo == null) {
            currentGeo = "0";
        }
        if (group.getAdgroupType() != null && Objects.equals(group.getAdgroupType(), MOBILE_CONTENT_GROUP)) {
            devices = Arrays.asList(DEVICE_PHONE, DEVICE_TABLET);
        } else {
            devices = Collections.emptyList();
        }
    }

    /**
     * Обработать строку из таблицы с ключевыми фразами. Предполагается, что вся информация о группе и кампании у нас уже есть
     */
    private void processBid(YtBidsRow bid) {
        String originalKeyword = bid.getPhrase();
        if (originalKeyword == null || "".equals(originalKeyword)) {
            return;
        }

        OfflineAdvqExportOutputTableRow output = new OfflineAdvqExportOutputTableRow();
        output.setId(bid.getId());
        output.setGroupId(bid.getPid());
        output.setOriginalKeyword(originalKeyword);
        output.setDevices(devices);
        output.setGeo(currentGeo);

        Set<String> minusKeywords = new HashSet<>(groupMinusPhrases);
        String keyword = extractKeywordAndMinusKeywords(originalKeyword, minusKeywords);
        output.setKeyword(keyword);
        try {
            output.setMinusWords(getMinusWordsWithoutIntersections(keyword, minusKeywords));
        } catch (IllegalQueryException | UnsupportedOperationException | UncheckedExecutionException e) {
            System.err.println(String.format("%s %s %s %s %s", bid.getId(), bid.getPid(),
                    Collections.singletonList(originalKeyword), groupMinusPhrases, e.getMessage()));
        }

        this.yield(output);
    }

    /**
     * Получить плюс-часть фразы и добавить минус-часть в список минус-фраз.
     * ВАЖНО: minusKeywords изменяется внутри метода
     *
     * @param originalKeyword оригинальная плюс-фраза из базы данных
     * @param minusKeywords   оригинальное множество минус фраз, включая минус-фразы на группу
     */
    String extractKeywordAndMinusKeywords(String originalKeyword, Set<String> minusKeywords) {
        // Это должно корректно работать на фразах из bids
        String[] data = SPLIT_PATTERN.split(originalKeyword);
        String keyword = "";
        if (data.length > 0) {
            keyword = data[0];
            minusKeywords.addAll(cleanMinusWords(Arrays.asList(data).subList(1, data.length)));
        }
        return keyword;
    }

    /**
     * Получить подмножество минус фраз, которые не пересекаются с плюс-фразой
     *
     * @param keyword       плюс-фраза
     * @param minusKeywords оригинальное множество минус фраз, включая минус-фразы на группу и минус-слова из самой фразы
     */
    List<String> getMinusWordsWithoutIntersections(String keyword,
                                                   Collection<String> minusKeywords) {
        if (keyword.startsWith("\"") || minusKeywords.isEmpty()) {
            return Collections.emptyList();
        }
        Collection<String> intersection =
                getIncludedMinusKeywords(keywordFactory, stopWords, Collections.singletonList(keyword), minusKeywords);
        if (intersection.isEmpty()) {
            return new ArrayList<>(minusKeywords);
        } else {
            Set<String> intersectionWords = new HashSet<>(intersection);
            return minusKeywords.stream().filter(w -> !intersectionWords.contains(w)).collect(Collectors.toList());
        }
    }

    private static Set<String> getIncludedMinusKeywords(KeywordWithLemmasFactory keywordFactory, Set<String> stopWords,
                                                        Collection<String> plusKeywords, Collection<String> minusKeywords) {
        return KeywordInclusionUtils
                .getIncludedMinusKeywords(keywordFactory, stopWords::contains, plusKeywords, minusKeywords);
    }
}
