package ru.yandex.direct.jobs.advq.offline.processing;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YqlQuery;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.direct.ytwrapper.specs.OperationSpec;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;

import static ru.yandex.direct.jobs.configuration.JobsEssentialConfiguration.DEFAULT_YT_CLUSTER;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;

/**
 * Джоба, которая подготавливает данные со свежими прогнозами показов из ADVQ:
 * - отсеивает те прогнозы, в которых не изменилось количество показов (или оно изменилось менее чем на 0,5%)
 * - делит данные по шардам и раскладывает каждый шард в отдельную таблицу в YT-кластере
 * <p>
 * ВАЖНО: при запуске на тестовых средах возможны ошибки из-за того, что количество шардов на них может отличаться от
 * количества шардов в проде, а данные в YT всегда продовые
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 10),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_1, CheckTag.YT, CheckTag.GROUP_INTERNAL_SYSTEMS}
)
@Hourglass(cronExpression = "5 53 */3 * * ?", needSchedule = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class OfflineAdvqProcessingJob extends DirectJob {
    public static final String UPLOAD_TIME_PROP_NAME =
            String.format("%s_last_upload_time", OfflineAdvqProcessingJob.class.getSimpleName());
    public static final String FORECAST_TIME_PROP_NAME =
            String.format("%s_last_forecast_time", OfflineAdvqProcessingJob.class.getSimpleName());
    static final String FORECAST_TIME_TABLE_ATTR = "dates_range";
    private static final Logger logger = LoggerFactory.getLogger(OfflineAdvqProcessingJob.class);

    private static final String YQL_QUERY =
            new ClassPathResourceInputStreamSource("advq/offline/processing/pre_processing.yql").readText();

    private final YtProvider ytProvider;
    private final YtCluster defaultYtCluster;
    private final ShardHelper shardHelper;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    private YtOperator ytOperator;
    private YtClusterConfig ytClusterConfig;

    @Autowired
    public OfflineAdvqProcessingJob(YtProvider ytProvider,
                                    @Qualifier(DEFAULT_YT_CLUSTER) YtCluster defaultYtCluster,
                                    ShardHelper shardHelper,
                                    PpcPropertiesSupport ppcPropertiesSupport) {
        this.ytProvider = ytProvider;
        this.defaultYtCluster = defaultYtCluster;
        this.shardHelper = shardHelper;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    @Override
    public void execute() {
        ytClusterConfig = ytProvider.getClusterConfig(defaultYtCluster);
        ytOperator = ytProvider.getOperator(defaultYtCluster);

        YtTable inputTable = new YtTable(OfflineAdvqProcessingBaseTableRow.INPUT_PATH);
        String forecastTime = getForecastTime(ytOperator
                .readTableStringAttribute(inputTable,
                        FORECAST_TIME_TABLE_ATTR));
        String uploadTime = ytOperator.readTableUploadTime(inputTable);
        if (!canRun(uploadTime)) {
            logger.info("ADVQ forecast table is already processed");
            return;
        }
        YtTable pidToShardTable = runPidToShardMapReduce();

        // На всякий случай запросим дату обновления еещ раз, вдруг она поменялась
        uploadTime = ytOperator.readTableUploadTime(inputTable);
        runMapReduce(pidToShardTable);

        setProcessedTableProps(uploadTime, forecastTime);
    }

    String getForecastTime(String period) {
        String date = Iterables.getLast(Arrays.asList(period.split("-")));

        return LocalDate.parse(date, DateTimeFormatter.ofPattern("yyyyMMdd")).atStartOfDay().plusDays(1).toString();
    }

    boolean canRun(String uploadTime) {
        String processedUploadTime = ppcPropertiesSupport.get(UPLOAD_TIME_PROP_NAME);
        logger.info("{} property value is '{}'. Upload time is {}", UPLOAD_TIME_PROP_NAME, processedUploadTime,
                uploadTime);
        return processedUploadTime == null || !processedUploadTime.equals(uploadTime);
    }

    private void setProcessedTableProps(String uploadTime, String forecastTime) {
        logger.info("Setting {} property value to '{}'", UPLOAD_TIME_PROP_NAME, uploadTime);
        ppcPropertiesSupport.set(UPLOAD_TIME_PROP_NAME, uploadTime);

        logger.info("Setting {} property value to '{}'", FORECAST_TIME_PROP_NAME, forecastTime);
        ppcPropertiesSupport.set(FORECAST_TIME_PROP_NAME, forecastTime);
    }

    /**
     * Запустить предварительную задачу в MR-кластере. Предварительная задача собирает данные о соответствии
     * идентификаторов групп их шардам в базе Директа
     */
    private YtTable runPidToShardMapReduce() {
        String shardTablePath = YtPathUtil.generateTemporaryPath();

        logger.info("Starting yql prep job");
        ytOperator.yqlExecute(new YqlQuery(YQL_QUERY, shardTablePath)
                .withTitle("OfflineAdvqProcessingJob:pre_processing"));
        logger.info("Prep yql finished successfully: {}", shardTablePath);

        return new YtTable(shardTablePath);
    }

    /**
     * Запустить задачу подготовки данных к импорту в MR-кластере
     *
     * @param pidToShardTable объект таблицы с данными о шардах групп
     */
    private void runMapReduce(YtTable pidToShardTable) {
        logger.info("Preparing map-reduce job");
        OperationSpec spec =
                OfflineAdvqProcessingMRSpec.getSpec(ytClusterConfig.getHome(), pidToShardTable, shardHelper.dbShards());
        logger.info("Starting map-reduce job");
        ytOperator.runOperation(spec);
        logger.info("Map-reduce finished successfully");
    }
}
