package ru.yandex.direct.jobs.advq.offline.processing;

import java.time.Duration;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;

import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.direct.ytwrapper.model.YtTableRow;
import ru.yandex.direct.ytwrapper.specs.AppendableSpecBuilder;
import ru.yandex.direct.ytwrapper.specs.MapReduceSpecBuilder;
import ru.yandex.direct.ytwrapper.specs.OperationSpec;
import ru.yandex.direct.ytwrapper.specs.SortSpecBuilder;
import ru.yandex.direct.ytwrapper.tables.generated.YtBids;
import ru.yandex.direct.ytwrapper.tables.generated.YtBidsRow;
import ru.yandex.direct.ytwrapper.tables.generated.YtDbTables;
import ru.yandex.direct.ytwrapper.tables.generated.YtPhrases;
import ru.yandex.direct.ytwrapper.tables.generated.YtPhrasesRow;
import ru.yandex.misc.dataSize.DataSize;

import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.ytwrapper.specs.AppendableSpecBuilder.pair;

/**
 * Спецификация основной MR-таски джобы
 */
@ParametersAreNonnullByDefault
public class OfflineAdvqProcessingMRSpec {
    private static final Duration MR_TIMEOUT = Duration.ofHours(3);
    private static final String OUTPUT_PATH_BASE = "import/offline_advq/phrases_forecast";

    private static final List<AppendableSpecBuilder.TableRowPair> MR_TABLES =
            Collections.unmodifiableList(Arrays.asList(
                    pair(new YtTable(""), new OfflineAdvqPreProcessingResultRow()),
                    pair(YtDbTables.PHRASES, new YtPhrasesRow(Arrays.asList(
                            YtPhrases.PID,
                            YtPhrases.GEO))),
                    pair(YtDbTables.BIDS, new YtBidsRow(Arrays.asList(
                            YtBids.PID,
                            YtBids.ID,
                            YtBids.PHRASE,
                            YtBids.SHOWSFORECAST))),
                    pair(new YtTable(OfflineAdvqProcessingBaseTableRow.INPUT_PATH),
                            new OfflineAdvqProcessingBaseTableRow())
            ));

    private OfflineAdvqProcessingMRSpec() {
    }

    /**
     * Получить список описаний рядов таблиц из которых получается информация для экспорта фраз
     */
    static List<YtTableRow> getMRTablesRows() {
        return mapList(MR_TABLES, AppendableSpecBuilder.TableRowPair::getRow);
    }

    /**
     * Создать и заполнить билдер спецификации MR-задачи
     *
     * @param outputHome путь к домашней папке, в которой будет работать задача
     * @param preTable   таблица, из которой можно брать соответствие идентификатора группы шарду
     * @param shards     список шардов в БД директа. Для каждого шарда будет создана отдельная таблица с прогнозами фраз
     *                   из него
     */
    static OperationSpec getSpec(String outputHome, YtTable preTable, Collection<Integer> shards) {
        List<AppendableSpecBuilder.TableRowPair> tables = ImmutableList.<AppendableSpecBuilder.TableRowPair>builder()
                .add(pair(preTable, MR_TABLES.get(0).getRow()))
                .addAll(MR_TABLES.subList(1, MR_TABLES.size()))
                .build();

        MapReduceSpecBuilder mapReduceSpecBuilder = new MapReduceSpecBuilder()
                .addInputTables(tables)
                .setMapJobCount(4096)
                .setMapper(new OfflineAdvqProcessingMapper())
                .addReduceByField(YtPhrases.PID)
                .addSortByField(YtPhrases.PID)
                .addSortByField(YtBids.ID)
                .addSortByField(YtTableRow.TI_FIELD)
                .setReducer(new OfflineAdvqProcessingReducer())
                .setReducerMemoryLimit(DataSize.fromMegaBytes(1500))
                .setPartitionCount(4096)
                .setPartitionJobCount(4096);

        for (Integer shard : shards) {
            YtTable tmpShardTable = new YtTable(YtPathUtil.generateTemporaryPath());
            mapReduceSpecBuilder.addOutputTable(tmpShardTable);

            new SortSpecBuilder()
                    .addSortByField(OfflineAdvqProcessingBaseTableRow.GROUP_ID)
                    .addSortByField(OfflineAdvqProcessingBaseTableRow.ID)
                    .addOutputTable(new YtTable(getShardTableName(outputHome, shard)))
                    .addInputTable(tmpShardTable)
                    .appendTo(mapReduceSpecBuilder);
        }

        return mapReduceSpecBuilder
                .setOperationTimeout(MR_TIMEOUT)
                .build();
    }

    /**
     * Получить путь к таблице с прогнозами показов отдельного шарда
     *
     * @param outputHome путь, относительно которого строятся все пути
     * @param shard      номер шарда в Директе
     */
    public static String getShardTableName(String outputHome, int shard) {
        return YtPathUtil.generatePath(outputHome, relativePart(),
                OUTPUT_PATH_BASE + "_" + String.format("shard_%s", shard));
    }
}
