package ru.yandex.direct.jobs.advq.offline.processing;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.ytwrapper.model.YtReducer;
import ru.yandex.direct.ytwrapper.model.YtTableRow;
import ru.yandex.direct.ytwrapper.model.YtYield;
import ru.yandex.direct.ytwrapper.tables.generated.YtBidsRow;
import ru.yandex.direct.ytwrapper.tables.generated.YtPhrasesRow;

/**
 * Reduce-операция, которая:
 * - отсеивает те прогнозы, в которых не изменилось количество показов (или оно изменилось менее чем на 0,5%)
 * - делит данные по шардам и раскладывает каждый шард в отдельную таблицу в YT-кластере
 * <p>
 * Предполагает, что входящие данные отсортированы по идентификатору группы
 * <p>
 * Работает следующим образом:
 * 1. Собирает всю доступную информацию о группе: ее номер, геотаргетинг, шард, исторические прогнозы фраз и сами фразы,
 * а также данные о новых прогнозах фраз
 * 2. Если по группе недостает какой-то информации, пропускает ее
 * 3. Если текст фразы не изменился, а значение прогноза изменилось более чем на 0,5% от предыдущего,
 * передает ряд данных с описанием на вывод нужного шарда
 * 4. Если в группе не изменился прогноз ни по одной фразе, передает технический ряд данных в таблицу шарда для того,
 * чтобы скрипт импорта значений мог проставить новую дату пересчета прогноза в базе.
 * <p>
 * Технический ряд данных содержит идентификатор и геотаргетинг группы а также нулевые и пустые значения во всех остальных
 * полях вывода
 */
@ParametersAreNonnullByDefault
public class OfflineAdvqProcessingReducer extends YtReducer<Long> {
    // если новое значение отличается от старого меньше чем на 5% или меньше
    // чем на 10 хитов - не обновляем значения - бережём нашу базу
    static final double BORDER_PERCENT_VALUE = 5.0;
    static final long BORDER_DIFF_VALUE = 10;

    private final List<YtTableRow> rowsOrder = OfflineAdvqProcessingMRSpec.getMRTablesRows();

    @SuppressWarnings("WeakerAccess")
    public OfflineAdvqProcessingReducer() {
        super();
    }

    @SuppressWarnings("WeakerAccess")
    public OfflineAdvqProcessingReducer(YtYield ytYield) {
        super(ytYield);
    }

    @Override
    public Long key(YtTableRow row) {
        return row.valueOf(OfflineAdvqProcessingTemporaryTableRow.PID);
    }

    @Override
    public void reduce(Long groupId, Iterator<YtTableRow> entries) {
        Long shard = null;
        String geo = null;
        Map<Long, String> idToOldPhrase = new HashMap<>();
        Map<Long, Long> idToOldForecast = new HashMap<>();
        Map<Long, OfflineAdvqProcessingBaseTableRow> idToNewRow = new HashMap<>();
        while (entries.hasNext()) {
            YtTableRow row = entries.next();
            int tableIndex = row.getTableIndex();

            YtTableRow realRow = rowsOrder.get(tableIndex);
            realRow.setDataFrom(row);

            if (realRow instanceof OfflineAdvqPreProcessingResultRow) {
                shard = ((OfflineAdvqPreProcessingResultRow) realRow).getShard();
            } else if (realRow instanceof YtPhrasesRow) {
                YtPhrasesRow phrasesRow = (YtPhrasesRow) realRow;
                geo = phrasesRow.getGeo();
            } else if (realRow instanceof YtBidsRow) {
                YtBidsRow bidsRow = (YtBidsRow) realRow;
                idToOldPhrase.put(bidsRow.getId(), bidsRow.getPhrase());
                idToOldForecast.put(bidsRow.getId(), bidsRow.getShowsForecast());
            } else if (realRow instanceof OfflineAdvqProcessingBaseTableRow) {
                OfflineAdvqProcessingBaseTableRow importRow = (OfflineAdvqProcessingBaseTableRow) realRow;

                OfflineAdvqProcessingBaseTableRow newRow = new OfflineAdvqProcessingBaseTableRow();
                newRow.setId(importRow.getId());
                newRow.setGroupId(groupId);
                newRow.setGeo(importRow.getGeo());
                newRow.setOriginalKeyword(importRow.getOriginalKeyword());
                newRow.setForecast(importRow.getForecast());

                idToNewRow.put(newRow.getId(), newRow);
            } else {
                throw new RuntimeException("Unexpected table");
            }
        }

        yieldRowsIfNessesary(shard, groupId, geo, idToOldPhrase, idToOldForecast, idToNewRow);
    }

    private void yieldRowsIfNessesary(@Nullable Long shard, Long groupId, @Nullable String geo,
                                      Map<Long, String> idToOldPhrase, Map<Long, Long> idToOldForecast,
                                      Map<Long, OfflineAdvqProcessingBaseTableRow> idToNewRow) {
        // Если не хватает данных из одной из таблиц, пропускаем группу
        if (geo == null || shard == null || idToNewRow.isEmpty() || idToOldPhrase.isEmpty()) {
            return;
        }

        boolean dataDiffers = false;
        List<OfflineAdvqProcessingBaseTableRow> rowsToYield = new ArrayList<>();
        for (OfflineAdvqProcessingBaseTableRow row : idToNewRow.values()) {
            if (!row.getGeo().equals(geo) ||
                    !row.getOriginalKeyword().equals(idToOldPhrase.get(row.getId())) ||
                    idToOldForecast.get(row.getId()) == null ||
                    row.getForecast() == null) {
                dataDiffers = true;
                continue;
            }

            if (forecastDiffers(row.getForecast(), idToOldForecast.get(row.getId()))) {
                rowsToYield.add(row);
            }
        }


        if (rowsToYield.isEmpty() && !dataDiffers) {
            OfflineAdvqProcessingBaseTableRow groupRow = new OfflineAdvqProcessingBaseTableRow();
            groupRow.setId(0L);
            groupRow.setGroupId(groupId);
            groupRow.setGeo(geo);
            groupRow.setOriginalKeyword("");
            groupRow.setForecast(0L);

            rowsToYield.add(groupRow);
        }

        final int yieldShard = shard.intValue();
        rowsToYield.forEach(r -> this.yield(yieldShard - 1, r));
    }

    /**
     * Посчитать изменение значения прогноза в целых процентах от предыдущего значения.
     *
     * @param newForecast новый прогноз показов
     * @param oldForecast старый прогноз показов
     */
    double percentChange(long newForecast, long oldForecast) {
        if (newForecast == oldForecast) {
            return 0;
        } else if (oldForecast == 0) {
            return 100;
        }
        return Math.abs((newForecast - oldForecast * 1.0) / oldForecast) * 100.0;
    }

    boolean forecastDiffers(long newForecast, long oldForecast) {
        return Math.abs(newForecast - oldForecast) > BORDER_DIFF_VALUE
                && percentChange(newForecast, oldForecast) > BORDER_PERCENT_VALUE;
    }
}
