package ru.yandex.direct.jobs.agencyclientprove;

import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.client.model.AgencyClientProve;
import ru.yandex.direct.core.entity.client.repository.AgencyClientProveRepository;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Очищает таблицу {@link ru.yandex.direct.dbschema.ppc.tables.AgencyClientProve} от запросов на подтверждение,
 * которые были добавлены раньше, чем {@code BORDER_DAYS} = {@value BORDER_DAYS} дней назад
 */
@ParametersAreNonnullByDefault
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2, hours = 2),
        tags = {DIRECT_PRIORITY_2, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION})
@Hourglass(cronExpression = "0 30 4 * * ?", needSchedule = TypicalEnvironment.class)
public class AgencyClientProveCleaner extends DirectShardedJob {
    private static final int BORDER_DAYS = 30;

    private static final Logger logger = LoggerFactory.getLogger(AgencyClientProveCleaner.class);

    private final AgencyClientProveRepository agencyClientProveRepository;

    @Autowired
    public AgencyClientProveCleaner(AgencyClientProveRepository agencyClientProveRepository) {
        this.agencyClientProveRepository = agencyClientProveRepository;
    }

    @Override
    public void execute() {
        LocalDateTime borderDateTime = LocalDateTime.now().minusDays(BORDER_DAYS);

        logger.info("get outdated prove requests using border date {}", borderDateTime);
        List<AgencyClientProve> outdatedRequests =
                agencyClientProveRepository.getOutdatedRequests(getShard(), borderDateTime);
        logger.info("got {} requests to delete", outdatedRequests.size());

        for (AgencyClientProve request : outdatedRequests) {
            logger.info("request to delete: {}", JsonUtils.toJson(request));
            agencyClientProveRepository.deleteRequest(getShard(), request);
        }
    }
}
