package ru.yandex.direct.jobs.agencyofflinereport;

import java.net.URL;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.DeleteObjectsResult;
import com.amazonaws.services.s3.model.ListObjectsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.env.TestingOnly;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Удаляет все старые объекты из бакета с отчетами. Это требуется из-за переналивок бд директа, а квота в MDS конечная.
 * Только для тестового MDS-S3. Если потребуется в продакшене - сначала нужно договориться с mds-admin@.
 * <p>
 * Удаляются отчеты старше {@code agency_offline_report.report_lifetime} + {@link #LIFETIME_GAP}
 * <p>
 * Удаление реализовано путем цикличного листинга всего бакета целиком, из каждого чанка - удаляются старые объекты.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 3, hours = 3),
        tags = {DIRECT_PRIORITY_2, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION},
        needCheck = TestingOnly.class
)
@Hourglass(cronExpression = "40 40 6 * * ?", needSchedule = TestingOnly.class)
@ParametersAreNonnullByDefault
public class AbandonedAgencyOfflineReportCleaner extends DirectJob {
    private static final Duration LIFETIME_GAP = Duration.ofDays(5);

    private static final Logger logger = LoggerFactory.getLogger(AbandonedAgencyOfflineReportCleaner.class);

    private final AmazonS3 amazonS3;
    private final String bucketName;
    private final Duration reportLifetime;

    @Autowired
    public AbandonedAgencyOfflineReportCleaner(AmazonS3 amazonS3, DirectConfig directConfig) {
        this.amazonS3 = amazonS3;

        DirectConfig config = directConfig.getBranch("agency_offline_report");
        bucketName = config.getString("bucket_name");
        checkArgument(!bucketName.isEmpty(), "S3 bucket name can't be empty");
        reportLifetime = config.getDuration("report_lifetime").plus(LIFETIME_GAP);
    }

    @Override
    public void execute() {
        Date borderDate = Date.from(Instant.now().minus(reportLifetime).truncatedTo(ChronoUnit.SECONDS));

        logger.debug("Search abandoned reports older than {}", borderDate);

        ListObjectsRequest listRequest = new ListObjectsRequest().withBucketName(bucketName).withMaxKeys(1);
        ObjectListing listResult = new ObjectListing();
        // ставим признак "не все объекты попали в ответ" чтобы первый раз зайти в цикл
        listResult.setTruncated(true);

        int iteration = 0;
        while (listResult.isTruncated()) {
            logger.debug("fetch objects chunk #{}", ++iteration);
            listResult = amazonS3.listObjects(listRequest);
            List<DeleteObjectsRequest.KeyVersion> keyVersions = listResult.getObjectSummaries()
                    .stream()
                    .filter(o -> o.getLastModified().before(borderDate))
                    .peek(this::logDelete)
                    .map(S3ObjectSummary::getKey)
                    .map(DeleteObjectsRequest.KeyVersion::new)
                    .collect(Collectors.toList());

            DeleteObjectsRequest deleteRequest = new DeleteObjectsRequest(bucketName).withKeys(keyVersions)
                    .withQuiet(true);
            amazonS3.deleteObjects(deleteRequest).getDeletedObjects().forEach(this::logDeleteFailed);
            listRequest = listRequest.withMarker(listResult.getNextMarker());
        }
    }

    private void logDelete(S3ObjectSummary summary) {
        URL url = amazonS3.getUrl(bucketName, summary.getKey());
        logger.info("Going to delete from S3: {} (lastModified {})", url, summary.getLastModified());
    }

    private void logDeleteFailed(DeleteObjectsResult.DeletedObject object) {
        logger.error("Failed to delete object {} from S3", amazonS3.getUrl(bucketName, object.getKey()));
    }
}
