package ru.yandex.direct.jobs.agencyofflinereport;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.sql.Connection;
import java.sql.SQLException;
import java.time.Duration;
import java.time.format.DateTimeFormatter;

import javax.annotation.ParametersAreNonnullByDefault;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.google.common.base.Joiner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.util.LocaleGuard;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.agencyofflinereport.container.AgencyOfflineReportBuilderJobResult;
import ru.yandex.direct.core.entity.agencyofflinereport.container.AgencyOfflineReportJobParams;
import ru.yandex.direct.core.entity.agencyofflinereport.model.AgencyOfflineReportKind;
import ru.yandex.direct.core.entity.agencyofflinereport.service.AgencyOfflineReportService;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbqueue.JobFailedPermanentlyException;
import ru.yandex.direct.dbqueue.model.DbQueueJob;
import ru.yandex.direct.dbqueue.service.DbQueueService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.yql.YqlPreparedStatementImpl;
import ru.yandex.yql.response.QueryFileDto;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.core.entity.agencyofflinereport.service.AgencyOfflineReportParametersService.AGENCY_STAT_START_DATE;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;
import static ru.yandex.direct.juggler.check.model.CheckTag.YT;

/**
 * Строит Excel-отчет на основе заявок из {@link DbQueueService}. Отчет строится на основе кубов, хранимых в YT
 * и поддерживаемых отделом бизнес-аналитики. Названия столбцов переводятся на язык для уведомлений, заданный
 * в настройках пользователя, заказавшего отчет.
 * <p>
 * Типичное время построения отчета ~ 8 минут, возможны пики в час-полтора. Это без учета времени нахождения в очереди.
 * <a href="https://ppcgraphite.yandex.ru/grafana/dashboard/db/direct-agency-offline-report?orgId=1">Дашборд по очереди отчетов</a>
 * <p>
 * Для изменения выгрузки нужно:<ul>
 * <li>отредактировать yql-файл</li>
 * <li>добавить в {@link ReportColumnNameTranslations} название для новых столбцов</li>
 * <li>добавить в {@link AgencyOfflineReportExcelFactory#buildColumnTranslations()} связь "имени столбца в yql-выгрузке"
 * и добавленного в предыдущем пункте перевода.</li>
 * <li>там же зафиксирован порядок столбцов в отчете (совпадает с указанным)</li>
 * </ul>
 * <p>
 * Готовые отчеты выгружаются в MDS-S3, срок хранения задан в конфигурации {@code agency_offline_report.report_lifetime}
 * Графики использования S3:<ul>
 * <li>ref="https://yasm.yandex-team.ru/template/panel/s3_client/bucket=direct-agency-offline-reports;
 * owner=169/?range=86400000">в продакшн</a>
 * </li>
 * <li>ref="https://yasm.yandex-team.ru/template/panel/s3_client/bucket=direct-agency-offline-reports;owner=169;
 * ctype=testing/?range=86400000">в тестинге</a>
 * </li>
 * </ul>
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 6),
        tags = {DIRECT_PRIORITY_1, GROUP_INTERNAL_SYSTEMS, YT, JOBS_RELEASE_REGRESSION})
@Hourglass(periodInSeconds = 155, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class AgencyOfflineReportBuilder extends DirectShardedJob {
    private static final int MAX_ATTEMPTS = 5;
    private static final Duration GRAB_FOR = Duration.ofMinutes(180);

    static final String CLIENT_CUBES_PREFIX = "//statbox/cube/daily/comdep/hypercubes/v1/clients";
    static final DateTimeFormatter CUBE_DATES_FORMATTER = DateTimeFormatter.ISO_LOCAL_DATE;
    static final YtCluster CLIENT_CUBES_YT_CLUSTER = YtCluster.HAHN;

    private static final Logger logger = LoggerFactory.getLogger(AgencyOfflineReportBuilder.class);

    private static final Joiner JOINER = Joiner.on("\n");

    private final YtProvider ytProvider;
    private final YtCluster ytCluster;
    private final String clientsQuery;
    private final DbQueueService dbQueueService;
    private final UserService userService;
    private final ClientService clientService;
    private final String bucketName;
    private final double yqlQueryPriority;
    private final String ytPool;
    private final AmazonS3 amazonS3;
    private final AgencyOfflineReportService agencyOfflineReportService;
    private final AgencyOfflineReportExcelFactory excelRendererFactory;

    @Autowired
    public AgencyOfflineReportBuilder(YtProvider ytProvider,
                                      AmazonS3 amazonS3,
                                      DbQueueService dbQueueService,
                                      UserService userService,
                                      ClientService clientService,
                                      DirectConfig directConfig,
                                      AgencyOfflineReportExcelFactory excelRendererFactory,
                                      AgencyOfflineReportService agencyOfflineReportService) {
        this.ytProvider = ytProvider;
        this.amazonS3 = amazonS3;
        this.dbQueueService = dbQueueService;
        this.userService = userService;
        this.clientService = clientService;
        this.excelRendererFactory = excelRendererFactory;
        this.agencyOfflineReportService = agencyOfflineReportService;

        DirectConfig config = directConfig.getBranch("agency_offline_report");
        yqlQueryPriority = config.getDouble("yql_query_priority");
        checkArgument(yqlQueryPriority > 0, "yqlQueryPriority must be positive");
        bucketName = config.getString("bucket_name");
        checkArgument(!bucketName.isEmpty(), "S3 bucket name can't be empty");

        this.clientsQuery = LiveResourceFactory
                .get("classpath:///agencyofflinereport/client_details_report.yql").getContent();

        this.ytCluster = CLIENT_CUBES_YT_CLUSTER;

        String ytPool = config.getString("yt_pool");
        if (ytPool.isEmpty()) {
            // если пул не задан в настройках - используем пул-имени-пользователя
            ytPool = ytProvider.getClusterConfig(this.ytCluster).getUser();
        }
        this.ytPool = ytPool;
    }


    @Override
    public void execute() {
        dbQueueService.grabAndProcessJob(getShard(), DbQueueJobTypes.AGENCY_DASHBOARD_REPORT, GRAB_FOR,
                this::processGrabbedJob, MAX_ATTEMPTS, this::handleProcessingException);
    }

    /**
     * Обработчик задачи на построение отчета
     *
     * @param jobInfo информация о задаче
     * @return результат выполнения задачи с результирующей ссылкой на отчет
     */
    private AgencyOfflineReportBuilderJobResult processGrabbedJob(
            DbQueueJob<AgencyOfflineReportJobParams, AgencyOfflineReportBuilderJobResult> jobInfo) {
        AgencyOfflineReportJobParams jobArgs = jobInfo.getArgs();
        logger.info("Start processing reportId {}, jobId {} (attempt {})", jobArgs.getReportId(),
                jobInfo.getId(), jobInfo.getTryCount());

        agencyOfflineReportService.markNewReportAsProcessing(jobArgs.getAgencyClientId(), jobArgs.getReportId());

        String stringFrom = CUBE_DATES_FORMATTER.format(jobArgs.getDateFrom());
        String stringTo = CUBE_DATES_FORMATTER.format(jobArgs.getDateTo());

        boolean renderAgencyAverageRow = doRenderAgencyAverageRow(jobArgs);

        ClientId agencyId = jobArgs.getAgencyClientId();
        Client agency = clientService.getClient(agencyId);
        CurrencyCode agencyWorkCurrency = agency.getWorkCurrency();
        if (agencyWorkCurrency == CurrencyCode.YND_FIXED) {
            agencyWorkCurrency = CurrencyCode.RUB;
        }
        AgencyOfflineReportHeader reportHeader = new AgencyOfflineReportHeader(agencyId.asLong(), agency.getName(),
                agencyWorkCurrency, stringFrom, stringTo);
        String balanceCurrencyName = agencyWorkCurrency.getCurrency().getBalanceCurrencyName();

        int agencyLimit = renderAgencyAverageRow ? 1 : 0;

        YPath endTablePath = YPath.simple(YtPathUtil.generatePath(CLIENT_CUBES_PREFIX, stringTo));
        if (!ytProvider.get(ytCluster).cypress().exists(endTablePath)) {
            String error = "Report source table doesn't exists: " + endTablePath;
            logger.error(error);
            throw new JobFailedPermanentlyException(error);
        }

        Language reportLanguage = checkNotNull(userService.getUser(jobInfo.getUid()), "report owner user").getLang();

        String reportUrl;
        try (LocaleGuard ignored = LocaleGuard.fromLanguage(reportLanguage);
             Connection conn = ytProvider.getYql(ytCluster, YtSQLSyntaxVersion.SQLv1).getConnection();
             YqlPreparedStatementImpl pstmt = (YqlPreparedStatementImpl) conn.prepareStatement(clientsQuery)) {
            pstmt.attachFile("client_id_list.txt", QueryFileDto.Type.CONTENT, JOINER.join(jobArgs.getSubclients()));

            int pIdx = 0;
            pstmt.setString(++pIdx, Double.toString(yqlQueryPriority));
            pstmt.setString(++pIdx, ytPool);
            pstmt.setString(++pIdx, stringFrom);
            pstmt.setString(++pIdx, stringTo);
            pstmt.setLong(++pIdx, agencyId.asLong());
            pstmt.setInt(++pIdx, agencyLimit);
            pstmt.setString(++pIdx, balanceCurrencyName);

            byte[] result;
            TraceProfile yqlOperationProfile = Trace.current().profile("AgencyOfflineReportBuilder:execute_yql_query");
            try (AgencyOfflineReportExcelFactory.Renderer renderer =
                         excelRendererFactory.getRenderer(pstmt.executeQuery(), jobArgs.getSubclients(), reportHeader,
                                 renderAgencyAverageRow)) {
                yqlOperationProfile.close();
                TraceProfile renderXlsProfile = Trace.current().profile("AgencyOfflineReportBuilder:render_report");
                result = renderer.render();
                renderXlsProfile.close();
            }
            reportUrl = uploadReportToMdsS3(result, jobArgs);
            agencyOfflineReportService
                    .markProcessingReportAsReady(jobArgs.getAgencyClientId(), jobArgs.getReportId(), reportUrl);
        } catch (SQLException | IOException ex) {
            throw new AgencyOfflineReportException(ex);
        }

        logger.info("Finish processing reportId {}, jobId {} (attempt {})", jobArgs.getReportId(),
                jobInfo.getId(), jobInfo.getTryCount());
        return AgencyOfflineReportBuilderJobResult.success(reportUrl);
    }

    /**
     * Обработчик последней (когда повтором больше не будет) ошибки выполнения dbqueue-задачи.
     * Помечает отчет в очереди как неуспешный
     *
     * @param jobInfo    информация о самой задаче
     * @param stacktrace текст выпавшего исключений, приведшего к фейлу
     * @return результат выполнения задачи с ошибкой
     */
    private AgencyOfflineReportBuilderJobResult handleProcessingException(
            DbQueueJob<AgencyOfflineReportJobParams, AgencyOfflineReportBuilderJobResult> jobInfo, String stacktrace) {
        agencyOfflineReportService
                .markReportFailed(jobInfo.getArgs().getAgencyClientId(), jobInfo.getArgs().getReportId());
        return AgencyOfflineReportBuilderJobResult.error(stacktrace);
    }

    /**
     * Загрузить отчет в MDS-S3
     *
     * @param reportBytes данные отчета
     * @param params      параметры отчета, по ним определяется имя файла и объекта в S3
     * @return ссылка на загруженный файл
     * @throws IOException при ошибке закрытия {@link ByteArrayInputStream}, т.е. никогда
     */
    private String uploadReportToMdsS3(byte[] reportBytes, AgencyOfflineReportJobParams params) throws IOException {

        String externalName = getFilename(params);
        String internalName = getObjectName(params);
        try (ByteArrayInputStream is = new ByteArrayInputStream(reportBytes);
             TraceProfile ignored = Trace.current().profile("AgencyOfflineReportBuilder:uploadReportToMdsS3");) {
            ObjectMetadata metadata = new ObjectMetadata();
            // NB: вот здесь хрупкое место: от попадания русских букв в имя файла - ломается подпись запроса к S3
            metadata.setContentDisposition(String.format("attachment; filename=\"%s\"", externalName));
            metadata.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
            metadata.setContentLength(reportBytes.length);
            /*
             * NB: если здесь будут выставлены какие-то заголовки для внутреннего использования,
             * то нужно будет добавить их фильтрацию в конфигурации nginx
             */
            logger.debug("Upload report {} to S3", params.getReportId());
            amazonS3.putObject(bucketName, internalName, is, metadata);
            String downloadUrl = amazonS3.getUrl(bucketName, internalName).toString();
            logger.debug("uploaded file to MDS S3, download url is {}", downloadUrl);
            return downloadUrl;
        }
    }

    /**
     * Получить имя файла для отчета
     *
     * @param params параметры построения отчета
     * @return строка с именем файла
     */
    private static String getFilename(AgencyOfflineReportJobParams params) {
        return String.format("%s-%s-%s-kpi-report-%s.xlsx",
                DateTimeFormatter.BASIC_ISO_DATE.format(params.getDateFrom()),
                DateTimeFormatter.BASIC_ISO_DATE.format(params.getDateTo()),
                params.getAgencyLogin(),
                params.getReportKind().toString().toLowerCase());
    }

    /**
     * Получить имя файла для объекта S3
     *
     * @param params параметры построения отчета
     * @return строка с именем объекта
     */
    private static String getObjectName(AgencyOfflineReportJobParams params) {
        return String.format("%010d-%s", params.getReportId(), getFilename(params));
    }

    /**
     * Добавлять ли строчку со средним по агенству.
     * Добавляем, если отчет полный и дата больше чем доступная для построения
     *
     * @param jobArgs аргументы джобы
     * @return нужно ли добавлять строку
     */
    private static boolean doRenderAgencyAverageRow(AgencyOfflineReportJobParams jobArgs) {
        return jobArgs.getReportKind().equals(AgencyOfflineReportKind.FULL)
                && jobArgs.getDateFrom().compareTo(AGENCY_STAT_START_DATE) >= 0;
    }
}
