package ru.yandex.direct.jobs.agencyofflinereport;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3URI;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.DeleteObjectsResult;
import com.google.common.collect.Iterators;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.agencyofflinereport.model.AgencyOfflineReport;
import ru.yandex.direct.core.entity.agencyofflinereport.repository.AgencyOfflineReportRepository;
import ru.yandex.direct.dbschema.ppc.Tables;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Удаление устаревших агентских offline-отчетов:
 * <ul>
 * <li>метаинформация об отчетах - таблица Директа {@link Tables#AGENCY_OFFLINE_REPORTS}
 * <li>файлы отчетов - объекты в MDS-S3.
 * </ul>
 * Время жизни отчетов задается в конфигурации {@code agency_offline_report.report_lifetime}
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 3, hours = 3),
        tags = {DIRECT_PRIORITY_2, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION})
@Hourglass(cronExpression = "0 18 5 * * ?", needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class AgencyOfflineReportCleaner extends DirectShardedJob {
    /**
     * Предохранитель от ошибок в конфигурации
     */
    private static final int MIN_REPORT_LIFETIME_DAYS = 3;
    /**
     * Количество объектов, удаляемых за один запрос к S3
     */
    private static final int S3_DELETE_CHUNK_SIZE = 100;

    private static final Logger logger = LoggerFactory.getLogger(AgencyOfflineReportCleaner.class);

    private final AgencyOfflineReportRepository agencyOfflineReportRepository;
    private final AmazonS3 amazonS3;
    private final String bucketName;
    private final Duration reportLifetime;


    @Autowired
    public AgencyOfflineReportCleaner(AgencyOfflineReportRepository agencyOfflineReportRepository,
            AmazonS3 amazonS3, DirectConfig directConfig)
    {
        this.agencyOfflineReportRepository = agencyOfflineReportRepository;
        this.amazonS3 = amazonS3;

        DirectConfig config = directConfig.getBranch("agency_offline_report");
        bucketName = config.getString("bucket_name");
        checkArgument(!bucketName.isEmpty(), "S3 bucket name can't be empty");
        reportLifetime = config.getDuration("report_lifetime");
        checkArgument(reportLifetime.toDays() > MIN_REPORT_LIFETIME_DAYS,
                "Reports lifetime should be greater than %s days", MIN_REPORT_LIFETIME_DAYS);
    }

    @Override
    public void execute() {
        LocalDateTime border = LocalDateTime.now().minus(reportLifetime).truncatedTo(ChronoUnit.SECONDS);
        logger.debug("Going to delete reports older than {}", border);
        List<AgencyOfflineReport> reports = agencyOfflineReportRepository.getOutdatedReports(getShard(), border);
        logger.debug("got {} reports from db", reports.size());

        // Удаляем данные из S3 первыми, чтобы иметь возможность попробовать еще раз, если упадем
        // Временную недоступность отчетов по ссылке - считаем некритичной
        Iterator<DeleteObjectsRequest.KeyVersion> keyVersions = reports.stream()
                .map(AgencyOfflineReport::getReportUrl)
                .filter(Objects::nonNull)
                .peek(this::logDeleteFromS3)
                .map(AmazonS3URI::new)
                .map(AmazonS3URI::getKey)
                .map(DeleteObjectsRequest.KeyVersion::new).iterator();
        Iterators.partition(keyVersions, S3_DELETE_CHUNK_SIZE).forEachRemaining(this::deleteChunkFromS3);

        List<Long> reportIds = reports.stream().map(AgencyOfflineReport::getReportId).collect(Collectors.toList());
        int deleted = agencyOfflineReportRepository.deleteReports(getShard(), reportIds);
        logger.info("Deleted {} reports from DB", deleted);
    }

    private void logDeleteFromS3(String url) {
        logger.info("Going to delete from S3: {}", url);
    }

    private void logDeleteFromS3Failed(DeleteObjectsResult.DeletedObject object) {
        logger.error("Failed to delete object {} from S3", object.getKey());
    }

    private void deleteChunkFromS3(List<DeleteObjectsRequest.KeyVersion> chunk) {
        DeleteObjectsRequest request = new DeleteObjectsRequest(bucketName).withKeys(chunk).withQuiet(true);
        amazonS3.deleteObjects(request).getDeletedObjects().forEach(this::logDeleteFromS3Failed);
    }
}
