package ru.yandex.direct.jobs.agencyofflinereport;

import java.time.Duration;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes;
import ru.yandex.direct.dbqueue.LimitOffset;
import ru.yandex.direct.dbqueue.repository.DbQueueRepository;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Удаление устаревших задач (архивных и нет) по построению offline-отчетов.
 * Время жизни отчетов задается в конфигурации {@code agency_offline_report.report_lifetime}
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 3, hours = 3),
        tags = {DIRECT_PRIORITY_2, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION})
@Hourglass(cronExpression = "25 27 6 * * ?", needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class AgencyOfflineReportDbQueueCleaner extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(AgencyOfflineReportDbQueueCleaner.class);
    private static final LimitOffset LIMIT = LimitOffset.limited(10_000);

    private final DbQueueRepository dbQueueRepository;
    private final Duration reportLifetime;

    @Autowired
    public AgencyOfflineReportDbQueueCleaner(DbQueueRepository dbQueueRepository, DirectConfig directConfig) {
        this.dbQueueRepository = dbQueueRepository;

        reportLifetime = directConfig.getBranch("agency_offline_report").getDuration("report_lifetime");
    }

    @Override
    public void execute() {
        logger.debug("Going to delete {} jobs older than {}", LIMIT.limit(), reportLifetime);
        int r = dbQueueRepository.cleanup(getShard(), DbQueueJobTypes.AGENCY_DASHBOARD_REPORT, reportLifetime, LIMIT);
        logger.info("Deleted {} jobs", r);
    }
}
