package ru.yandex.direct.jobs.agencyofflinereport;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableSet;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.i18n.Translatable;

import static com.google.common.base.Preconditions.checkState;

/**
 * Фабрика классов для построения XLSX-отчета. Содержит в себе порядок столбцов и их переводы, позволяет получить
 * инстанс построителя по результату YQL-запроса и белому списку ClientID.
 */
@Service
class AgencyOfflineReportExcelFactory {

    /**
     * Мапинг "имя колонки в выгрузке" - "перевод"
     *
     * @implNote реализован на базе {@link LinkedHashMap} - для сохранения порядка колонка,
     * которая обернута в {@link Collections#unmodifiableMap(Map)} для консистентности
     */
    private static final Map<String, Translatable> columnTranslations = buildColumnTranslations();

    private static final String CLIENT_ID_COLUMN = "client_id";
    private static final String AGENCY_ID_COLUMN = "agency_id";
    private static final String ROW_TYPE_COLUMN = "row_type";
    private static final String AGENCY_ROW_TYPE = "aggregated_agency";
    private static final String CLIENT_ROW_TYPE = "client";
    private static final Set<String> utilColumns = ImmutableSet.of(ROW_TYPE_COLUMN);
    private static final Set<String> costColumns = ImmutableSet.of("cost", "search_cost", "network_cost", "smart_cost",
            "dynamic_cost", "mobile_content_cost", "mcb_cost", "cpm_cost", "desktop_cost", "mobile_cost",
            "drf_cost", "mobile_search_cost");

    private final TranslationService translationService;

    @Autowired
    public AgencyOfflineReportExcelFactory(TranslationService translationService) {

        this.translationService = translationService;
    }

    /**
     * Получить инстанс рендерера по ResultSet YQL-запроса и коллекции разрешенных ClientID.
     *
     * @param resultSet              результат выполнения YQL-запроса
     * @param allowedClientIds       коллекция разрешенных (к попаданию в отчет) ClientID
     * @param reportHeader           общие данные по агентству и датам составления отчета
     * @param renderAgencyAverageRow добавлять ли строку с средним по агенству
     * @return инстанс рендерера, готовый к построению отчета
     */
    public Renderer getRenderer(ResultSet resultSet, Collection<ClientId> allowedClientIds,
                                AgencyOfflineReportHeader reportHeader, boolean renderAgencyAverageRow) {
        return new Renderer(allowedClientIds, reportHeader, renderAgencyAverageRow, resultSet);
    }


    /**
     * Try-with-resources класс для разового превращения ResultSet в массив байт итогового XLS-отчета.
     */
    public class Renderer implements AutoCloseable {
        private final Logger logger = LoggerFactory.getLogger(Renderer.class);

        private final Set<Long> allowedClientIds;
        private final AgencyOfflineReportHeader reportHeader;
        private final Long reportAgencyId;
        private final boolean renderAgencyAverageRow;
        private final ResultSet rs;
        private final List<ColumnMetaInfo> columnsMetaInfo;
        private int currentRow;
        private final SXSSFWorkbook wb;
        private final Sheet sheet;
        private boolean canBeRendered;

        private final String averageByAgency = translationService.translate(
                ReportHeadersTranslations.INSTANCE.averageByAgency());

        private Renderer(Collection<ClientId> allowedClientIds, AgencyOfflineReportHeader reportHeader,
                         boolean renderAgencyAverageRow, ResultSet rs) {
            this.allowedClientIds = allowedClientIds.stream()
                    .map(ClientId::asLong)
                    .collect(ImmutableSet.toImmutableSet());
            this.reportHeader = reportHeader;
            this.reportAgencyId = reportHeader.getAgencyId();
            this.renderAgencyAverageRow = renderAgencyAverageRow;

            this.rs = rs;
            this.currentRow = 0;
            this.wb = new SXSSFWorkbook();
            this.sheet = wb.createSheet();
            this.canBeRendered = true;
            try {
                this.columnsMetaInfo = buildColumnMetadata(rs.getMetaData());
            } catch (SQLException e) {
                throw new IllegalStateException(e);
            }
        }


        /**
         * Создать отчет в формате Open Office XML Spreadsheet (XLSX) на основе {@link #rs} выполненного YQL-запроса.
         * Строки, значение столбца {@link #CLIENT_ID_COLUMN} которых не входит в {@link #allowedClientIds} - будут
         * пропущены (т.е. не попадут в итоговый отчет), значение client_id будет залогировано.
         *
         * @return результирующий XLS-файл в виде массива байт
         * @throws IOException           при ошибке ввода/вывода
         * @throws SQLException          при ошибках доступа к данным в {@code rs}
         * @throws IllegalStateException при повторном вызове метода на данном инстансе класса
         */
        public byte[] render() throws IOException, SQLException {
            checkState(canBeRendered, "Report already rendered");
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            renderTableHeader();
            while (rs.next()) {
                long agencyId = rs.getLong(AGENCY_ID_COLUMN);

                if (agencyId != reportAgencyId) {
                    logger.error("prohibited AgencyId {} was seen within ResultSet; filtered out", agencyId);
                    continue;
                }

                String rowType = rs.getString(ROW_TYPE_COLUMN);

                if (AGENCY_ROW_TYPE.equals(rowType) && renderAgencyAverageRow) {
                    // добавляем раздел "Среднее по агенству"
                    renderRowWithString("");
                    renderRowWithString(averageByAgency);
                    Row row = sheet.createRow(currentRow);
                    for (ColumnMetaInfo colInfo : columnsMetaInfo) {
                        renderCell(row, colInfo);
                    }
                    // убираем 0 у clientId для красоты
                    row.getCell(0).setCellValue("");
                    currentRow++;
                }


                if (CLIENT_ROW_TYPE.equals(rowType)) {

                    long clientId = rs.getLong(CLIENT_ID_COLUMN);

                    if (allowedClientIds.contains(clientId)) {
                        Row row = sheet.createRow(currentRow);
                        for (ColumnMetaInfo colInfo : columnsMetaInfo) {
                            renderCell(row, colInfo);
                        }
                        currentRow++;
                    } else {
                        logger.error("prohibited ClientId {} was seen within ResultSet; filtered out", clientId);
                    }
                }
            }
            wb.write(os);
            canBeRendered = false;
            return os.toByteArray();
        }


        /**
         * Собрать метаданные о столбцах по метаданным ответа.
         * Логирует предупреждением столбцы, описанные в {@link #columnTranslations} но отсутствующие в ResultSet
         *
         * @param metaData метаданные о ResultSet ответа
         * @return список {@link ColumnMetaInfo}
         * @throws SQLException при ошибке типизации
         */
        private List<ColumnMetaInfo> buildColumnMetadata(ResultSetMetaData metaData) throws SQLException {
            List<ColumnMetaInfo> columnsMetaInfo = new ArrayList<>(metaData.getColumnCount());
            Map<String, Integer> colTypesIndex = buildColumnTypesIndex(metaData);
            int colXlsReportIndex = 0;
            for (Map.Entry<String, Translatable> entry : columnTranslations.entrySet()) {
                String rsColumnLabel = entry.getKey();
                if (colTypesIndex.containsKey(rsColumnLabel)) {
                    ColumnMetaInfo metaInfo = new ColumnMetaInfo(colTypesIndex.get(rsColumnLabel), rsColumnLabel,
                            colXlsReportIndex, translationService.translate(entry.getValue()));
                    ++colXlsReportIndex;
                    columnsMetaInfo.add(metaInfo);
                } else {
                    logger.warn("column label '{}' found in translations, but not found in ResultSet", rsColumnLabel);
                }
            }

            return columnsMetaInfo;
        }


        /**
         * Построить словарь "имя столбца" -> "тип столбца" ({@link java.sql.Types})
         *
         * @param metaData метаданные о ResultSet ответа
         * @return новый словарь имя->тип
         * @throws SQLException          при ошибке типизации
         * @throws IllegalStateException при наличии в ResultSet столбцов, не описанных в {@link #columnTranslations}
         */
        private Map<String, Integer> buildColumnTypesIndex(ResultSetMetaData metaData) throws SQLException {
            int colCount = metaData.getColumnCount();
            Map<String, Integer> result = new HashMap<>(colCount);
            for (int colNum = 1; colNum <= colCount; colNum++) {
                String columnLabel = metaData.getColumnLabel(colNum);
                if (!utilColumns.contains(columnLabel)) {
                    checkState(columnTranslations.containsKey(columnLabel),
                            "column label '%s' in result set must have translation", columnLabel);
                    result.put(columnLabel, metaData.getColumnType(colNum));
                }
            }
            return result;
        }

        /**
         * Выставить тип и заполнить ячейку {@code cell} из {@code rs}
         *
         * @param row        строка в которой заполняем ячейки
         * @param columnInfo метаинформация о заполняемой ячейке
         * @throws SQLException при ошибке типизации читаемых из {@code rs} данных
         */
        private void renderCell(Row row, ColumnMetaInfo columnInfo) throws SQLException {
            Cell cell = row.createCell(columnInfo.getXlsReportIndex());
            String columnName = columnInfo.getRsColumnName();
            Object columnValue = rs.getObject(columnName);
            if (columnValue instanceof Long || columnValue instanceof Integer) {
                long longValue = ((Number) columnValue).longValue();
                cell.setCellType(CellType.NUMERIC);
                cell.setCellValue(longValue);
            } else if (columnValue instanceof Double || columnValue instanceof Float) {
                double doubleValue = ((Number) columnValue).doubleValue();
                if (costColumns.contains(columnName)) {
                    doubleValue = BigDecimal.valueOf(doubleValue)
                            .setScale(2, RoundingMode.HALF_UP)
                            .doubleValue();
                }
                cell.setCellType(CellType.NUMERIC);
                cell.setCellValue(doubleValue);
            } else if (columnInfo.getJdbcType() == Types.LONGVARBINARY
                    || columnInfo.getJdbcType() == Types.LONGNVARCHAR) {
                /*
                 * Особенность реализации YQL jdbc-драйвера: он преобразует нативный тип String в LONGVARBINARY,
                 * а utf8 - в LONGNVARCHAR. Приходится подстраиваться и использовать getString(),
                 * т.к. в случае с LONGVARBINARY getObject вернет массив.
                 */
                cell.setCellType(CellType.STRING);
                cell.setCellValue(rs.getString(columnName));
            } else {
                String strValue = columnValue == null ? null : columnValue.toString();
                cell.setCellType(CellType.STRING);
                cell.setCellValue(strValue);
            }
        }

        /**
         * Добавить на лист строку с заголовком
         */
        private void renderTableHeader() {
            ReportHeadersTranslations headerTranslations = ReportHeadersTranslations.INSTANCE;
            renderRowWithString(translationService.translate(headerTranslations.agencyReportHeader()));
            renderRowWithString(translationService.translate(
                    headerTranslations.agencyNameAndId(reportHeader.getAgencyName(), reportHeader.getAgencyId())));
            renderRowWithString(translationService.translate(
                    headerTranslations.reportDatesFromTo(reportHeader.getDateFrom(), reportHeader.getDateTo())));
            renderRowWithString("("
                    + translationService.translate(headerTranslations.agencyCurrency())
                    + ", "
                    + translationService.translate(reportHeader.getAgencyWorkCurrency().getTranslation().longForm())
                    + ")");
            renderRowWithString("");

            Row row = sheet.createRow(currentRow);
            for (ColumnMetaInfo colInfo : columnsMetaInfo) {
                Cell cell = row.createCell(colInfo.getXlsReportIndex());
                cell.setCellValue(colInfo.getHumanReadableName());
            }
            currentRow++;
        }

        private void renderRowWithString(String string) {
            Row row = sheet.createRow(currentRow++);
            row.createCell(0).setCellValue(string);
        }

        /**
         * Закрыть ResultSet и XLS-книгу
         *
         * @throws SQLException при ошибке закрытия {@link #rs}
         * @throws IOException  при ошибке закрытия книги
         */
        @Override
        public void close() throws SQLException, IOException {
            rs.close();
            wb.dispose();
            wb.close();
        }
    }


    /**
     * Создать мапинг "имя столбца" - "перевод".
     * В отчете столбцы будут строго в перечисленном здесь порядке.
     * <p>
     * Закоментированный в данный момент строки - фактически отсутствуют в отчете.
     *
     * @return заполненный и защищенный от изменений инстанс {@link LinkedHashMap}
     */
    private static Map<String, Translatable> buildColumnTranslations() {
        ReportColumnNameTranslations translations = ReportColumnNameTranslations.INSTANCE;
        LinkedHashMap<String, Translatable> map = new LinkedHashMap<>();

        map.put(CLIENT_ID_COLUMN, translations.clientId());
        map.put("client_login", translations.clientLogin());
        map.put("client_city", translations.clientCity());
        map.put("client_create_date", translations.clientCreateDate());

        map.put(AGENCY_ID_COLUMN, translations.agencyId());
        map.put("agency_name", translations.agencyName());
        map.put("agency_city", translations.agencyCity());
        map.put("agency_create_date", translations.agencyCreateDate());
        map.put("representative_login", translations.representativeLogin());

        map.put("shows", translations.impressions());
        map.put("search_shows", translations.searchImpressions());
        map.put("network_shows", translations.networkImpressions());
        map.put("clicks", translations.clicks());
        map.put("search_clicks", translations.searchClicks());
        map.put("network_clicks", translations.networkClicks());

        map.put("cost", translations.cost());
        map.put("search_cost", translations.searchCost());
        map.put("network_cost", translations.networkCost());
        map.put("smart_cost", translations.smartBannersCost());
        map.put("dynamic_cost", translations.dynamicAdsCost());
        map.put("mobile_content_cost", translations.mobileAppAdsCost());
        map.put("cpm_cost", translations.displayCampaignsCost());
        map.put("mcb_cost", translations.searchBannerCost());
        map.put("desktop_cost", translations.desktopCost());
        map.put("mobile_cost", translations.mobileCost());
        map.put("drf_cost", translations.relevantKeywordsCost());

        map.put("ctr", translations.ctr());
        map.put("search_ctr", translations.searchCtr());
        map.put("network_ctr", translations.networkCtr());

        map.put("mobile_search_shows", translations.mobileSearchImpressions());
        map.put("mobile_search_clicks", translations.mobileSearchClicks());
        map.put("mobile_search_cost", translations.mobileSearchCost());
        map.put("mobile_search_ctr", translations.mobileSearchCtr());

        map.put("client_has_stops", translations.numberOfCliensWithStops());
        map.put("client_max_stop", translations.maxClientIdleInDays());
        map.put("client_has_wallet", translations.numberOfClientsHasWallet());

        map.put("orders", translations.averageCampaignsCount());
        map.put("text_orders", translations.averageTextAndImageAdsCampaignsCount());
        map.put("smart_orders", translations.averageSmartBannersCampaignsCount());
        map.put("dynamic_orders", translations.averageDynamicAdsCampaignsCount());
        map.put("mobile_content_orders", translations.averageMobileAppAdsCampaignsCount());
        map.put("mcb_orders", translations.averageSearchBannerCampaignsCount());
        map.put("cpm_orders", translations.averageDisplayCampaignsCount());
        map.put("search_enabled_nondynamic_orders", translations.averageSearchEnabledNonDynamicCampaignsCount());
        map.put("context_enabled_nondynamic_orders", translations.averageNetworkEnabledNonDynamicCampaignsCount());
        map.put("search_and_context_enabled_nondynamic_orders",
                translations.averageSearchAndNetworkEnabledNonDynamicCampaignsCount());
        map.put("context_only_nondynamic_orders", translations.averageNetworkOnlyNonDynamicCampaignsCount());
        map.put("autobudget_orders", translations.averageNumberOfCampaignsWithAutobudget());
        map.put("goal_orders", translations.averageNumberOfCampaignsWithGoal());
        map.put("drf_orders", translations.averageNumberOfCampaignsWithRelevantKeywords());
        map.put("timetarget_orders", translations.averageNumberOfCampaignsWithTimetarget());
        map.put("metrika_orders", translations.averageNumberOfCampaignsWithMetrika());
        map.put("openstat_enabled_orders", translations.averageNumberOfCampaignsWithOpenstatEnabled());

        map.put("groups", translations.averageNumberOfAdGroups());
        map.put("text_groups", translations.averageNumberOfTextAndImageAdsAdGroups());
        map.put("context_enabled_text_groups",
                translations.averageNumberOfAdGroupsWithinTextAndImageAdsCampaignsWithNetworkEnabled());
        map.put("context_enabled_mobile_content_groups",
                translations.averageNumberOfMobileAppAdGroupsWithNetworkEnabled());
        map.put("with_desktop_groups", translations.averageNumberOfAdGroupsWithDesktopBanners());
        map.put("with_mobile_groups", translations.averageNumberOfAdGroupsWithMobileBanners());
        map.put("retargeting_conditions_groups", translations.averageNumberOfAdGroupsWithRetargetingConditions());
        map.put("audience_retargeting_conditions_groups", translations.averageNumberOfAdGroupsWithAudienceConditions());
        map.put("mobile_increasing_groups", translations.averageNumberOfAdGroupsWithIncreasingMobileAdjustments());
        map.put("mobile_decreasing_groups", translations.averageNumberOfAdGroupsWithDecreasingMobileAdjustments());
        map.put("mobile_multipliers_groups", translations.averageNumberOfAdGroupsWithMobileAdjustments());
        map.put("retargeting_increasing_groups",
                translations.averageNumberOfAdGroupsWithIncreasingRetargetingAdjustments());
        map.put("retargeting_decreasing_groups",
                translations.averageNumberOfAdGroupsWithDecreasingRetargetingAdjustments());
        map.put("retargeting_multipliers_groups", translations.averageNumberOfAdGroupsWithRetargetingAdjustments());
        map.put("demography_increasing_groups",
                translations.averageNumberOfAdGroupsWithIncreasingDemographicsAdjustments());
        map.put("demography_decreasing_groups",
                translations.averageNumberOfAdGroupsWithDecreasingDemographicsAdjustments());
        map.put("demography_multipliers_groups", translations.averageNumberOfAdGroupsWithDemographicsAdjustments());
        map.put("with_image_ad_groups", translations.averageNumberOfAdGroupsWithImageAdAds());
        map.put("with_wide_image_groups", translations.averageNumberOfAdGroupsWithAdsWithWideImage());
        map.put("search_relevance_match_groups", translations.averageNumberOfAdGroupsWithAutotargeting());

        map.put("banners", translations.averageNumberOfAds());
        map.put("text_banners", translations.averageNumberOfTextAndImageAds());
        map.put("dynamic_banners", translations.averageNumberOfDynamicTextAds());
        map.put("mobile_content_banners", translations.averageNumberOfMobileAppAds());
        map.put("search_enabled_text_banners",
                translations.averageNumberOfTextAndImageAdsWithinCampaignsWithEnabledSearch());
        map.put("context_enabled_text_banners",
                translations.averageNumberOfTextAndImageAdsWithinCampaignsWithEnabledNetwork());
        map.put("context_enabled_mobile_content_banners",
                translations.averageNumberOfMobileAppAdsWithinCampaignsWithEnabledNetwork());
        map.put("sitelinks_banners", translations.averageNumberOfAdsWithSitelinks());
        map.put("sitelinks_with_description_banners", translations.averageNumberOfAdsWithSitelinksWithDescription());
        map.put("image_banners", translations.averageNumberOfAdsWithImage());
        map.put("display_href_banners", translations.averageNumberOfAdsWithDisplayUrl());
        map.put("callout_banners", translations.averageNumberOfAdsWithCallouts());
        map.put("vcard_banners", translations.averageNumberOfAdsWithVCard());
        map.put("title_extension_banners", translations.averageNumberOfAdsWithSecondTitle());
        map.put("moderation_rejected_banners", translations.averageNumberOfAdsRejectedOnModeration());

        map.put("text_orders_pct", translations.shareOfTextAndImageAdsCampaigns());
        map.put("smart_orders_pct", translations.shareOfSmartBannersCampaigns());
        map.put("dynamic_orders_pct", translations.shareOfDynamicAdsCampaigns());
        map.put("mobile_content_orders_pct", translations.shareOfMobileAppAdsCampaigns());
        map.put("mcb_orders_pct", translations.shareOfSearchBannerCampaigns());
        map.put("cpm_orders_pct", translations.shareOfDisplayCampaigns());
        map.put("search_enabled_orders_pct", translations.shareOfCampaignsWithSearchEnabled());
        map.put("context_enabled_orders_pct", translations.shareOfCampaignsWithNetworkEnabled());
        map.put("search_and_context_enabled_orders_pct", translations.shareOfCampaignsWithSearchAndNetworkEnabled());
        map.put("context_only_orders_pct", translations.shareOfCampaignsWithNetworkOnly());
        map.put("autobudget_orders_pct", translations.shareOfCampaignsWithAutobudget());
        map.put("goal_orders_pct", translations.shareOfCampaignsWithGoal());
        map.put("drf_orders_pct", translations.shareOfCampaignsWithRelevantKeywords());
        map.put("timetarget_orders_pct", translations.shareOfCampaignsWithTimetarget());
        map.put("metrika_orders_pct", translations.shareOfCampaignsWithMetrika());
        map.put("openstat_enabled_orders_pct", translations.shareOfCampaignsWithOpenstatEnabled());

        map.put("with_desktop_groups_pct", translations.shareOfAdGroupsWithDesktopAds());
        map.put("with_mobile_groups_pct", translations.shareOfAdGroupsWithMobileAds());
        map.put("retargeting_conditions_groups_pct", translations.shareOfAdGroupsWithRetargetingConditions());
        map.put("audience_retargeting_conditions_groups_pct", translations.shareOfAdGroupsWithAudienceConditions());
        map.put("mobile_increasing_groups_pct", translations.shareOfAdGroupsWithIncreasingMobileAdjustments());
        map.put("mobile_decreasing_groups_pct", translations.shareOfAdGroupsWithDecreasingMobileAdjustments());
        map.put("mobile_multipliers_groups_pct", translations.shareOfAdGroupsWithMobileAdjustments());
        map.put("retargeting_increasing_groups_pct",
                translations.shareOfAdGroupsWithIncreasingRetargetingAdjustments());
        map.put("retargeting_decreasing_groups_pct",
                translations.shareOfAdGroupsWithDecreasingRetargetingAdjustments());
        map.put("retargeting_multipliers_groups_pct", translations.shareOfAdGroupsWithRetargetingAdjustments());
        map.put("demography_increasing_groups_pct",
                translations.shareOfAdGroupsWithIncreasingDemographicsAdjustments());
        map.put("demography_decreasing_groups_pct",
                translations.shareOfAdGroupsWithDecreasingDemographicsAdjustments());
        map.put("demography_multipliers_groups_pct", translations.shareOfAdGroupsWithDemographicsAdjustments());
        map.put("with_image_ad_groups_pct", translations.shareOfAdGroupsWithImageAdAds());
        map.put("with_wide_image_groups_pct", translations.shareOfAdGroupsWithAdsWithWideImage());
        map.put("search_relevance_match_groups_pct", translations.shareOfAdGroupsWithAutotargeting());

        map.put("sitelinks_banners_pct", translations.shareOfAdsWithSitelinks());
        map.put("sitelinks_with_description_banners_pct", translations.shareOfAdsWithSitelinksWithDescription());
        map.put("image_banners_pct", translations.shareOfAdsWithImage());
        map.put("display_href_banners_pct", translations.shareOfAdsWithDisplayUrl());
        map.put("callout_banners_pct", translations.shareOfAdsWithCallouts());
        map.put("vcard_banners_pct", translations.shareOfAdsWithVCard());
        map.put("title_extension_banners_pct", translations.shareOfAdsWithSecondTitle());
        map.put("moderation_rejected_banners_pct", translations.shareOfAdsRejectedOnModeration());

        map.put("active_clients", translations.numberOfClientsActiveDuringPeriod());
        map.put("clients", translations.averageNumberOfClientsDuringPeriod());
        map.put("clients_with_stops_pct", translations.shareOfClientsWithStops());
        map.put("wallet_clients_pct", translations.shareOfClientsWithWallet());

        map.put("with_cpc_video_groups", translations.averageNumberOfAdGroupsWithVideoAds());
        map.put("video_addition_banners", translations.averageNumberOfBannersWithVideoAdditions());
        map.put("with_cpc_video_groups_pct", translations.shareOfAdGroupsWithVideoAds());
        map.put("video_addition_banners_pct", translations.shareOfBannersWithVideoAdditions());

        return Collections.unmodifiableMap(map);
    }
}
