package ru.yandex.direct.jobs.agencyofflinereport;

import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Lazy;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerEvent;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.DirectNumericCheck;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;

import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.YT;


/**
 * Числовая проверка отствания исходных данных по кубам, в часах.
 */
@Lazy
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 5),
        tags = {GROUP_INTERNAL_SYSTEMS, YT},
        needCheck = ProductionOnly.class,
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT})
)
@ParametersAreNonnullByDefault
public class LastAvailableDateCheck extends DirectNumericCheck {

    private static final Duration WARN_BORDER = Duration.ofDays(4);
    private static final Duration CRIT_BORDER = Duration.ofDays(8).plusHours(8);

    private static final String CHECK_NAME = "comdep-hypercubes-clients.last_available_date.lag";
    private static final String CHECK_DESCRIPTION = "Отставание последнего доступного гиперкуба по клиентам, в часах";

    @Autowired
    public LastAvailableDateCheck(@Value("${juggler.checks.target_host}") String host) {
        super(CHECK_NAME, WARN_BORDER.toHours(), CRIT_BORDER.toHours(), host, CHECK_DESCRIPTION, true);
    }

    public JugglerEvent generateEvent(LocalDate lastAvailableDate) {
        Duration lag = Duration.between(lastAvailableDate.atStartOfDay(), LocalDateTime.now());
        return getEvent(lag.toHours(), CHECK_NAME, null);
    }

    @Override
    public List<String> getRawServiceNames() {
        return Collections.singletonList(CHECK_NAME);
    }
}
