package ru.yandex.direct.jobs.agencyofflinereport;

import java.time.Duration;
import java.time.LocalDate;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.agencyofflinereport.service.AgencyOfflineReportParametersService;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerEvent;
import ru.yandex.direct.juggler.check.JugglerNumericEventsClient;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.inside.yt.kosher.cypress.YPath;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.jobs.agencyofflinereport.AgencyOfflineReportBuilder.CLIENT_CUBES_PREFIX;
import static ru.yandex.direct.jobs.agencyofflinereport.AgencyOfflineReportBuilder.CLIENT_CUBES_YT_CLUSTER;
import static ru.yandex.direct.jobs.agencyofflinereport.AgencyOfflineReportBuilder.CUBE_DATES_FORMATTER;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;
import static ru.yandex.direct.juggler.check.model.CheckTag.YT;


/**
 * Задача для обновления свойства {@link AgencyOfflineReportParametersService#MAX_REPORT_DATE_PROP_NAME}.
 * Записывает последнюю дату за которую есть данные, делает числовую juggler-проверку отставания этой даты.
 * <p>
 * Пологается на то, что таблицы идут без пропусков. Умеет искать таблицы вперед и назад,
 * имеет защиту от бесконечного поиска конечной таблицы.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 20),
        tags = {DIRECT_PRIORITY_1, GROUP_INTERNAL_SYSTEMS, YT, JOBS_RELEASE_REGRESSION})
@Hourglass(periodInSeconds = 3200, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class LastAvailableDateFetcher extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(LastAvailableDateFetcher.class);
    private static final int DEFAULT_MAX_STEPS = 45;
    private static final Duration EVENT_SEND_TIMEOUT = Duration.ofMinutes(10);

    private final AgencyOfflineReportParametersService parametersService;
    private final JugglerNumericEventsClient jugglerNumericEventsClient;
    private final LastAvailableDateCheck lastAvailableDateCheck;
    private final YtProvider ytProvider;
    private final int maxSteps;

    @Autowired
    public LastAvailableDateFetcher(AgencyOfflineReportParametersService parametersService,
                                    JugglerNumericEventsClient jugglerNumericEventsClient,
                                    LastAvailableDateCheck lastAvailableDateCheck,
                                    YtProvider ytProvider) {
        this(parametersService, jugglerNumericEventsClient, lastAvailableDateCheck, ytProvider, DEFAULT_MAX_STEPS);
    }

    LastAvailableDateFetcher(AgencyOfflineReportParametersService parametersService,
                             JugglerNumericEventsClient jugglerNumericEventsClient,
                             LastAvailableDateCheck lastAvailableDateCheck,
                             YtProvider ytProvider,
                             int maxSteps) {
        this.parametersService = parametersService;
        this.jugglerNumericEventsClient = jugglerNumericEventsClient;
        this.lastAvailableDateCheck = lastAvailableDateCheck;
        this.ytProvider = ytProvider;
        checkArgument(maxSteps > 0, "maxSteps should be greater than zero");
        this.maxSteps = maxSteps;
    }

    @Override
    public void execute() {
        LocalDate start = parametersService.getMaximumAvailableDate();
        performJugglerCheck(start);

        LocalDate candidate = searchLastDate(start);

        if (candidate == null) {
            logger.warn("No available date found");
            return;
        }

        if (!candidate.equals(start)) {
            logger.info("Update maximum available date to {}", candidate);
            parametersService.setMaximumAvailableDate(candidate);
            performJugglerCheck(candidate);
        }
    }

    LocalDate searchLastDate(LocalDate from) {
        LocalDate result = null;

        // Ищем даты вперед. Текущую тоже проверяем, так как в property может быть предполагаемая дата
        logger.debug("Search tables forward from {}", from);
        for (int i = 0; i <= maxSteps; i++) {
            LocalDate candidate = from.plusDays(i);
            if (existsTable(candidate)) {
                // нашли - запоминаем и пытаемся искать дальше
                result = candidate;
            } else {
                break;
            }
        }
        if (result == null) {
            logger.debug("Search tables backward");
            // Ищем даты назад, сюда попадаем если таблицы с текущей датой тоже нет
            for (int i = 1; i <= maxSteps; i++) {
                LocalDate candidate = from.minusDays(i);
                if (existsTable(candidate)) {
                    // берем первую же найденную
                    result = candidate;
                    break;
                }
            }
        }
        return result;
    }

    static YPath getPathForDate(LocalDate date) {
        return YPath.simple(YtPathUtil.generatePath(CLIENT_CUBES_PREFIX, CUBE_DATES_FORMATTER.format(date)));
    }

    private boolean existsTable(LocalDate date) {
        YPath path = getPathForDate(date);
        boolean result = ytProvider.get(CLIENT_CUBES_YT_CLUSTER).cypress().exists(path);
        logger.debug("Table {} exists: {}", path, result);
        return result;
    }

    private void performJugglerCheck(LocalDate date) {
        JugglerEvent event = lastAvailableDateCheck.generateEvent(date);
        logger.debug("Sending to juggler available date lag info: {}", event);
        jugglerNumericEventsClient.sendEvent(event, EVENT_SEND_TIMEOUT);
    }
}
