package ru.yandex.direct.jobs.aggrstatusresyncqueue;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.concurrent.atomic.AtomicLong;

import org.jooq.util.mysql.MySQLDSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.Url;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.HourglassDaemon;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.direct.dbschema.ppc.Tables.AGGR_STATUSES_RESYNC_QUEUE;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRODUCT_TEAM;
import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;

/**
 * Отправляет в соломон информацию о состоянии очереди aggr_statuses_resync_queue:
 * возраст самого старого объекта (в минутах), размер очереди.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10),
        tags = {DIRECT_PRODUCT_TEAM, DIRECT_PRIORITY_1_NOT_READY},
        needCheck = NonDevelopmentEnvironment.class,
        urls = {
                @Url(title = "Возраст очереди в минутах", url =
                        AggrStatusesResyncQueueMonitoringJob.URL_SOLOMON_MINUTES),
                @Url(title = "Размер очереди", url = AggrStatusesResyncQueueMonitoringJob.URL_SOLOMON_SIZE),
        }
)
@HourglassDaemon(runPeriod = 60)
@Hourglass(periodInSeconds = 60, needSchedule = ProductionOnly.class)
public class AggrStatusesResyncQueueMonitoringJob extends DirectShardedJob {

    private static final Logger logger = LoggerFactory.getLogger(AggrStatusesResyncQueueMonitoringJob.class);

    public static final String AGGR_STATUSES_RESYNC_QUEUE_NAME = "aggr_statuses_resync_queue";
    private static final String AGE_MINUTES = "age_minutes";
    private static final String QUEUE_SIZE = "queue_size";
    private static final long DEFAULT_THRESHOLD_MINUTES = 600L;

    static final String URL_SOLOMON_MINUTES = "https://solomon.yandex-team.ru/?project=direct&cluster=app_java-jobs" +
            "&service=java_jobs&l.sensor=age_minutes&l.host=CLUSTER&graph=auto&l.queue=" + AGGR_STATUSES_RESYNC_QUEUE_NAME +
            "&l.env=production&stack=false";

    static final String URL_SOLOMON_SIZE = "https://solomon.yandex-team.ru/?project=direct&cluster=app_java-jobs" +
            "&service=java_jobs&l.sensor=queue_size&l.host=CLUSTER&graph=auto&l.queue=" + AGGR_STATUSES_RESYNC_QUEUE_NAME +
            "&l.env=production&stack=false";

    private final DslContextProvider dslContextProvider;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    private PpcProperty<Long> limitMinutesProperty;
    private MetricRegistry metricRegistry;

    private final AtomicLong ageMinutes = new AtomicLong();
    private final AtomicLong queueSize = new AtomicLong();

    @Autowired
    public AggrStatusesResyncQueueMonitoringJob(DslContextProvider dslContextProvider,
                                                PpcPropertiesSupport ppcPropertiesSupport) {
        this.dslContextProvider = dslContextProvider;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    @Override
    public void execute() {
        this.metricRegistry = lazyInitMetricRegistry();

        limitMinutesProperty = ppcPropertiesSupport.get(PpcPropertyNames.AGGR_STATUSES_RESYNC_QUEUE_AGE_LIMIT_MINUTES);

        ageMinutes.set(getAgeMinutes());
        queueSize.set(getQueueSize());

        setStatus();
    }

    private MetricRegistry lazyInitMetricRegistry() {
        Labels solomonRegistryLabels =
                Labels.of("queue", AGGR_STATUSES_RESYNC_QUEUE_NAME, "shard", String.valueOf(getShard()));
        MetricRegistry metricRegistry = SOLOMON_REGISTRY.subRegistry(solomonRegistryLabels);
        metricRegistry.lazyGaugeInt64(AGE_MINUTES, ageMinutes::get);
        metricRegistry.lazyGaugeInt64(QUEUE_SIZE, queueSize::get);
        return metricRegistry;
    }

    private long getAgeMinutes() {
        LocalDateTime oldestRecord = dslContextProvider.ppc(getShard())
                .select(MySQLDSL.min(AGGR_STATUSES_RESYNC_QUEUE.QUEUE_TIME))
                .from(AGGR_STATUSES_RESYNC_QUEUE)
                .fetchOne()
                .value1();

        if (oldestRecord == null) {
            return 0L;
        }

        LocalDateTime now = LocalDateTime.now();

        long ageMinutes = ChronoUnit.MINUTES.between(oldestRecord, now);
        logger.info("Queue age: {} minutes", ageMinutes);
        return ageMinutes;
    }

    private long getQueueSize() {
        int queueSize = dslContextProvider.ppc(getShard())
                .selectCount()
                .from(AGGR_STATUSES_RESYNC_QUEUE)
                .fetchOne()
                .value1();

        if (queueSize > 0) {
            logger.info("Queue_is_not_empty");
        }

        logger.info("Queue size: {} records", queueSize);
        return queueSize;
    }

    private void setStatus() {
        Duration ageThreshold = Duration.ofMinutes(limitMinutesProperty.getOrDefault(DEFAULT_THRESHOLD_MINUTES));
        long ageThresholdMinutes = ageThreshold.toMinutes();
        long currentAge = ageMinutes.get();
        if (currentAge > ageThresholdMinutes) {
            String desc = String.format("Queue age exceed threshold. Current: %dm. Threshold: %dm",
                    currentAge, ageThresholdMinutes);
            logger.info("Queue_age_exceeds_the_limit");
            setJugglerStatus(JugglerStatus.CRIT, desc);
        } else {
            String desc = String.format("Queue age OK. Threshold: %dm", ageThresholdMinutes);
            setJugglerStatus(JugglerStatus.OK, desc);
        }
    }

    @Override
    public void finish() {
        metricRegistry.removeMetric(AGE_MINUTES, Labels.empty());
        metricRegistry.removeMetric(QUEUE_SIZE, Labels.empty());
    }

}
