package ru.yandex.direct.jobs.aggrstatusresyncqueue;

import java.time.LocalDateTime;
import java.time.ZoneOffset;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.aggregatedstatuses.service.AggregatedStatusesResyncQueueService;
import ru.yandex.direct.core.entity.aggregatedstatus.model.AggregatedEntityIdWithType;
import ru.yandex.direct.core.entity.aggregatedstatus.model.AggregatedStatusQueueEntityType;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.jobs.aggrstatusresyncqueue.repository.CampaignStatisticRepository;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static java.time.LocalDateTime.now;
import static ru.yandex.direct.common.db.PpcPropertyNames.AGGREGATED_STATUS_RESYNC_QUEUE_ENABLED;
import static ru.yandex.direct.common.db.PpcPropertyNames.AGGREGATED_STATUS_RESYNC_QUEUE_LAST_UPDATE_TIME;
import static ru.yandex.direct.common.db.PpcPropertyNames.AGGREGATED_STATUS_RESYNC_QUEUE_PREPARING_TABLES_ENABLED;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;
import static ru.yandex.direct.juggler.check.model.NotificationRecipient.LOGIN_SSDMITRIEV;
import static ru.yandex.direct.utils.DateTimeUtils.MSK;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * Выгружаем из таблицы DirectGridStat статистику начиная с UpdateTime >= lastModificationTime
 * Смотрим какие новые кампании появились в этот период времени в DirectGridStat, относительно прошлого запуска
 * Добавляем их в очередь для пересчета агрегированного статуса
 * <p>
 * Запускаем раз в два часа. Вероятно потребуется чаще -- суммарный дневной объем пересчета статусов не должен меняться
 * от изменения частоты запусков.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 4, minutes = 5),
        needCheck = ProductionOnly.class,
        notifications = {
                @OnChangeNotification(recipient = LOGIN_SSDMITRIEV,
                        status = {JugglerStatus.OK, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM),
        },
        tags = {DIRECT_PRIORITY_1_NOT_READY}
)
@Hourglass(periodInSeconds = 60 * 60 * 2, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class AggregatedStatusesResyncQueueJob extends DirectShardedJob {

    private static final Logger logger = LoggerFactory.getLogger(AggregatedStatusesResyncQueueJob.class);

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final CampaignStatisticRepository campaignStatisticRepository;
    private final AggregatedStatusesResyncQueueService aggregatedStatusesResyncQueueService;

    public AggregatedStatusesResyncQueueJob(PpcPropertiesSupport ppcPropertiesSupport,
                                            CampaignStatisticRepository campaignStatisticRepository,
                                            AggregatedStatusesResyncQueueService aggregatedStatusesResyncQueueService) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.campaignStatisticRepository = campaignStatisticRepository;
        this.aggregatedStatusesResyncQueueService = aggregatedStatusesResyncQueueService;
    }

    @Override
    public void execute() {
        logger.info("Start execute job");
        var isJobEnabled = ppcPropertiesSupport.get(AGGREGATED_STATUS_RESYNC_QUEUE_ENABLED);

        if (!isJobEnabled.getOrDefault(false)) {
            logger.info("Job is disabled");
            return;
        }

        var isTablesPreparingEnabled =
                ppcPropertiesSupport.get(AGGREGATED_STATUS_RESYNC_QUEUE_PREPARING_TABLES_ENABLED);

        if (isTablesPreparingEnabled.getOrDefault(true)) {
            campaignStatisticRepository.prepareProcessedTable(getShard());
        }

        LocalDateTime now = now();
        var lastModificationTimeProperty = ppcPropertiesSupport.get(AGGREGATED_STATUS_RESYNC_QUEUE_LAST_UPDATE_TIME);
        var lastModificationTime = lastModificationTimeProperty.getOrDefault(now.minusDays(1));

        var startOfDayInSeconds = now.toLocalDate().atStartOfDay(MSK).toEpochSecond();
        var statDateFromInSeconds = getStatDateFromInSeconds(startOfDayInSeconds, now, lastModificationTime);

        var campaignConversionCountInfoList = campaignStatisticRepository
                .getNewConversionsCountInfoByCampaignId(getShard(), statDateFromInSeconds);
        var campaignEntitiesToAdd = listToSet(campaignConversionCountInfoList,
                info -> new AggregatedEntityIdWithType()
                        .withEntityId(info.getCampaignId())
                        .withType(AggregatedStatusQueueEntityType.CAMPAIGN));

        aggregatedStatusesResyncQueueService.addEntitiesToAggregatedStatusesResyncQueue(getShard(),
                campaignEntitiesToAdd);

        campaignStatisticRepository.saveProcessingConversionsCountInfoToProcessed(getShard(), startOfDayInSeconds);
        lastModificationTimeProperty.set(now);
        logger.info("Finish execute job");
    }

    /**
     * В табличках со статистикой есть update_time -- фактически, это указание дня(день 21:00)
     * после которого в течении 24 часов совершилась конверсия.
     * Если запуск джобы первый с начала "дня" то нужно прогрузить данные конверсий и за предыдущий день.
     * Иначе можем упустить часть конверсий совершенных в конце прошлого дня
     */
    private long getStatDateFromInSeconds(long startOfDayInSeconds, LocalDateTime now,
                                          LocalDateTime lastModificationTime) {
        long startOfYesterdayInSeconds = now.minusDays(1).toLocalDate().atStartOfDay(MSK).toEpochSecond();
        return lastModificationTime.toEpochSecond(ZoneOffset.UTC) < startOfDayInSeconds ?
                startOfYesterdayInSeconds : startOfDayInSeconds;
    }
}
