package ru.yandex.direct.jobs.aggrstatusresyncqueue.repository

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Value
import org.springframework.stereotype.Repository
import ru.yandex.direct.config.DirectConfig
import ru.yandex.direct.jobs.aggrstatusresyncqueue.AggregatedStatusesResyncQueueJob
import ru.yandex.direct.liveresource.LiveResourceFactory
import ru.yandex.direct.ytcomponents.service.CampaignGridStatContextProvider
import ru.yandex.direct.ytwrapper.client.YtExecutionUtil
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.exceptions.RuntimeYqlException
import ru.yandex.direct.ytwrapper.model.*
import java.sql.ResultSet
import javax.annotation.ParametersAreNonnullByDefault

@Repository
@ParametersAreNonnullByDefault
class CampaignStatisticRepository(
        private val ytProvider: YtProvider,
        directConfig: DirectConfig,
        @Value("\${aggr_status_resync_queue.conversion_campaign_with_statistic.path_prefix}")
        private val pathPrefix: String,
        @Value("\${aggr_status_resync_queue.conversion_campaign_with_statistic.mysql_sync_path}")
        private val mysqlSyncPath: String,
) {

    private val clusters: List<YtCluster> = directConfig.getStringList(
            "aggr_status_resync_queue.conversion_campaign_with_statistic.clusters")
            .map { YtCluster.parse(it) }
            .toList()

    companion object {
        private const val CREATE_TABLE_QUERY_RESOURCE =
                "classpath:///aggrstatusresyncqueue/create_processed_table.yql"
        private const val PROCESSED_TABLE_PREFIX = "processed:"

        private const val GET_QUERY_RESOURCE =
                "classpath:///aggrstatusresyncqueue/campaign_with_conversion_strategy_with_new_stat.yql"
        private const val UPDATE_QUERY_RESOURCE =
                "classpath:///aggrstatusresyncqueue/add_new_campaign_with_conversion_strategy_with_stat.yql"
        private val EXPORT_ID = YtField("export_id", Long::class.java)
        private val CONVERSION_COUNT = YtField("conversion_count", Long::class.java)
        private val UPDATE_TIME = YtField("update_time_in_sec", Long::class.java)

        private val logger = LoggerFactory.getLogger(AggregatedStatusesResyncQueueJob::class.java)

    }

    /**
     * Создает таблицу processed:shard, если на кластерах её еще нет.
     * Нужно, чтобы не падать при первом запуске на кластере.
     */
    fun prepareProcessedTable(shard: Int) {
        val tablePath = pathPrefix + PROCESSED_TABLE_PREFIX + shard;
        val ytTable = YtTable(tablePath)
        val query = LiveResourceFactory.get(CREATE_TABLE_QUERY_RESOURCE).content
        clusters.stream()
                .filter { !ytProvider.getOperator(it).exists(ytTable) }
                .forEach {
                    try {
                        ytProvider.getOperator(it).yqlExecute(query, tablePath)
                    } catch (e: RuntimeYqlException) {
                        logger.error("Got exception on trying save result of campaign statistic processing", e)
                    }
                }
    }

    fun getNewConversionsCountInfoByCampaignId(shard: Int, from: Long): MutableList<CampaignConversionCountInfo> {
        val query = LiveResourceFactory.get(GET_QUERY_RESOURCE).content

        return YtExecutionUtil.executeWithFallback(
                clusters,
                ytProvider::getOperator,
                getNewConversionsCountInfoByCampaignId(query, shard, from))
    }

    private fun getNewConversionsCountInfoByCampaignId(query: String, shard: Int, from: Long)
            : (YtOperator) -> MutableList<CampaignConversionCountInfo> = {
        val result = arrayListOf<CampaignConversionCountInfo>()
        it.yqlQuery(query, ExportIdMapper(result), shard, pathPrefix, mysqlSyncPath, from)
        result
    }


    fun saveProcessingConversionsCountInfoToProcessed(shard: Int, startOfDayInSeconds: Long) {
        val query = LiveResourceFactory.get(UPDATE_QUERY_RESOURCE).content
        clusters.forEach {
            try {
                ytProvider.getOperator(it).yqlExecute(query, shard, pathPrefix, startOfDayInSeconds)
            } catch (e: RuntimeYqlException) {
                logger.error("Got exception on trying save result of campaign statistic processing", e)
            }
        }
    }

    /**
     * Маппер, который на самом деле consumer (в целях экономии памяти).
     * Складывает результат в мапу, которая задается извне и обновляется
     * при вызове [.mapRow].
     *
     * Сам [.mapRow] всегда возвращает `null`.
     */
    private class ExportIdMapper constructor(private val result: MutableList<CampaignConversionCountInfo>) : YqlRowMapper<Unit> {
        @Throws(Exception::class)
        override fun mapRow(rs: ResultSet) {
            val exportId = rs.getString(EXPORT_ID.name).toLong()
            val count = rs.getString(CONVERSION_COUNT.name).toLong()
            val updateTimeInSeconds = rs.getString(UPDATE_TIME.name).toLong()

            result.add(CampaignConversionCountInfo(exportId, count, updateTimeInSeconds))
        }
    }

    data class CampaignConversionCountInfo(
            val campaignId: Long,
            val count: Long,
            val updateTimeInSeconds: Long,
    )

}
