package ru.yandex.direct.jobs.autobudget;

import java.time.LocalDateTime;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.autobudget.repository.AutobudgetCpaAlertRepository;
import ru.yandex.direct.core.entity.autobudget.repository.AutobudgetHourlyAlertRepository;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static ru.yandex.direct.core.entity.autobudget.model.AutobudgetAlertProperties.CPA;
import static ru.yandex.direct.core.entity.autobudget.model.AutobudgetAlertProperties.HOURLY;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Удаляет неактуальные алерты, которые в течении некоторого времени не обновлялись.
 * Таблицы: {@link ru.yandex.direct.dbschema.ppc.tables.AutobudgetAlerts},
 * {@link ru.yandex.direct.dbschema.ppc.tables.AutobudgetCpaAlerts}
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 30),
        tags = {DIRECT_PRIORITY_1_NOT_READY, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION})
@Hourglass(periodInSeconds = 300)
@ParametersAreNonnullByDefault
public class OutdatedAutobudgetAlertsCleaner extends DirectShardedJob {

    private static final Logger logger = LoggerFactory.getLogger(OutdatedAutobudgetAlertsCleaner.class);

    private final AutobudgetHourlyAlertRepository hourlyAlertRepository;
    private final AutobudgetCpaAlertRepository cpaAlertRepository;

    @Autowired
    public OutdatedAutobudgetAlertsCleaner(
            AutobudgetHourlyAlertRepository hourlyAlertRepository,
            AutobudgetCpaAlertRepository cpaAlertRepository) {
        this.hourlyAlertRepository = hourlyAlertRepository;
        this.cpaAlertRepository = cpaAlertRepository;
    }

    /**
     * Нужен для тестов для проставления shard-а
     */
    OutdatedAutobudgetAlertsCleaner(int shard,
                                    AutobudgetHourlyAlertRepository hourlyAlertRepository,
                                    AutobudgetCpaAlertRepository cpaAlertRepository) {
        super(shard);
        this.hourlyAlertRepository = hourlyAlertRepository;
        this.cpaAlertRepository = cpaAlertRepository;
    }

    @Override
    public void execute() {
        LocalDateTime now = LocalDateTime.now();
        deleteOutdatedHourlyAlerts(now);
        deleteOutdatedCpaAlerts(now);
    }

    void deleteOutdatedHourlyAlerts(LocalDateTime now) {
        LocalDateTime borderDateTime = now.minus(HOURLY.getTtlForActiveAlerts());
        logger.debug("Going to delete all hourly autobudget alerts older than {}", borderDateTime);
        int deleted = hourlyAlertRepository.deleteActiveAlertsOlderThanDateTime(getShard(), borderDateTime);
        logger.info("deleted {} hourly alerts", deleted);
    }

    void deleteOutdatedCpaAlerts(LocalDateTime now) {
        LocalDateTime borderDateTime = now.minus(CPA.getTtlForActiveAlerts());
        logger.debug("Going to delete all {} autobudget CPA alerts older than {}", borderDateTime);
        int deleted = cpaAlertRepository.deleteActiveAlertsOlderThanDateTime(getShard(), borderDateTime);
        logger.info("deleted {} CPA-alerts", deleted);
    }
}
