package ru.yandex.direct.jobs.autobudget;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.common.db.PpcPropertyType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithDayBudget;
import ru.yandex.direct.core.entity.strategy.repository.StrategyModifyRepository;
import ru.yandex.direct.core.entity.strategy.repository.StrategyTypedRepository;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Сбрасывает количество изменений дневного ограничения бюджета (сейчас его можно изменять 3 раза в день) и время
 * остановки кампании по исчерпанию дневного бюджета в таблице {@link ru.yandex.direct.dbschema.ppc.tables.CampOptions}
 * <p>
 * Синхронизация выполняется только раз в день
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 12), tags = {DIRECT_PRIORITY_0, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION})
@Hourglass(cronExpression = "14 0 */2 * * ?")
@ParametersAreNonnullByDefault
public class UpdateDayBudgetJob extends DirectShardedJob {

    static final String LAST_RUN_DATE_PROPERTY = "UPDATE_DAY_BUDGET_LAST_RUN_DATE_%d";
    static final DateTimeFormatter DATE_FORMATTER_FOR_LAST_RUN_DATE_PROPERTY = DateTimeFormatter.BASIC_ISO_DATE;
    static final int UPDATE_PACKET_SIZE = 1_000;
    static final int DEFAULT_DAY_BUDGET_DAILY_CHANGE_COUNT = 0;

    private static final Logger logger = LoggerFactory.getLogger(UpdateDayBudgetJob.class);

    private final CampaignRepository campaignRepository;
    private final StrategyTypedRepository strategyTypedRepository;
    private final StrategyModifyRepository strategyModifyRepository;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    @Autowired
    public UpdateDayBudgetJob(CampaignRepository campaignRepository,
                              StrategyTypedRepository strategyTypedRepository,
                              StrategyModifyRepository strategyModifyRepository,
                              PpcPropertiesSupport ppcPropertiesSupport) {
        this.campaignRepository = campaignRepository;
        this.strategyTypedRepository = strategyTypedRepository;
        this.strategyModifyRepository = strategyModifyRepository;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    /**
     * Конструктор нужен для тестов для проставления shard-а
     */
    UpdateDayBudgetJob(int shard,
                       CampaignRepository campaignRepository,
                       StrategyTypedRepository strategyTypedRepository,
                       StrategyModifyRepository strategyModifyRepository,
                       PpcPropertiesSupport ppcPropertiesSupport) {
        super(shard);
        this.campaignRepository = campaignRepository;
        this.strategyTypedRepository = strategyTypedRepository;
        this.strategyModifyRepository = strategyModifyRepository;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    @Override
    public void execute() {
        if (!canRun()) {
            String message = "Skip iteration. Already update day budget today";
            logger.info(message);
            setJugglerStatus(JugglerStatus.OK, message);
            return;
        }

        LocalDateTime now = LocalDateTime.now();
        resetDayBudgetStopTime(now);
        resetDayBudgetChangeCount();

        updateLastRunProperty(now.toLocalDate());
    }

    boolean canRun() {
        LocalDate today = LocalDate.now();
        LocalDate lastRunDate = fetchLastRunDateProperty();
        return lastRunDate == null || lastRunDate.isBefore(today);
    }

    /**
     * Сбрасываем время остановки и факт отправки уведомления об остановке по дневному бюджету
     *
     * @param stopTime - время, которую проставляем в camp_options.day_budget_stop_time
     */
    void resetDayBudgetStopTime(LocalDateTime stopTime) {
        logger.debug("Fetching cids to reset day_budget_stop_time");
        Set<Long> cidsToResetStopTime =
                campaignRepository.getCampaignIdsWhichDayBudgetStopTimeLessThan(getShard(), stopTime);
        for (List<Long> chunk : Iterables.partition(cidsToResetStopTime, UPDATE_PACKET_SIZE)) {
            logger.info("Resetting day_budget_stop_time and notifications for cids: {}", chunk);
            campaignRepository.resetDayBudgetStopTimeAndNotificationStatus(getShard(), chunk, stopTime);
        }
    }

    /**
     * Сбрасываем количество изменений дневного бюджета
     */
    void resetDayBudgetChangeCount() {
        logger.debug("Fetching cids to reset day_budget_change_count");
        Set<Long> cidsToResetChangeCount = campaignRepository.getCampaignIdsWithDayBudgetDailyChanges(getShard());
        for (List<Long> chunk : Iterables.partition(cidsToResetChangeCount, UPDATE_PACKET_SIZE)) {
            logger.info("Resetting day_budget_change_count in cids: {}", chunk);
            campaignRepository.resetDayBudgetDailyChangeCount(getShard(), chunk);
            resetDayBudgetDailyChangeCountForStrategies(chunk);
        }
    }

    private void resetDayBudgetDailyChangeCountForStrategies(List<Long> campaignIds) {
        Map<Long, Long> strategyIdsByCampaignIds = campaignRepository.getStrategyIdsByCampaignIds(getShard(), campaignIds);

        Map<Long, StrategyWithDayBudget> strategyByStrategyIds =
                EntryStream.of(strategyTypedRepository.getIdToModelTyped(
                                getShard(),
                                strategyIdsByCampaignIds.values()))
                        .selectValues(StrategyWithDayBudget.class)
                        .toMap();

        List<AppliedChanges<StrategyWithDayBudget>> appliedChanges = StreamEx.of(campaignIds)
                .filter(strategyIdsByCampaignIds::containsKey)
                .map(strategyIdsByCampaignIds::get)
                .filter(strategyByStrategyIds::containsKey)
                .map(strategyId -> ModelChanges.build(
                        strategyId,
                        StrategyWithDayBudget.class,
                        StrategyWithDayBudget.DAY_BUDGET_DAILY_CHANGE_COUNT,
                        DEFAULT_DAY_BUDGET_DAILY_CHANGE_COUNT)
                )
                .map(modelChange -> modelChange.applyTo(strategyByStrategyIds.get(modelChange.getId())))
                .toList();

        if (!appliedChanges.isEmpty()) {
            strategyModifyRepository.updateStrategiesTable(getShard(), appliedChanges);
        }
    }

    void updateLastRunProperty(LocalDate today) {
        String lastRunDate = today.format(DATE_FORMATTER_FOR_LAST_RUN_DATE_PROPERTY);
        logger.info("Setting \"{}\" property to \"{}\"", getLastRunDatePropertyName(), lastRunDate);
        ppcPropertiesSupport.get(getLastRunDatePropertyName()).set(today);
    }

    @Nullable
    private LocalDate fetchLastRunDateProperty() {
        try {
            LocalDate updateDayBudgetLastRunDate = ppcPropertiesSupport.get(getLastRunDatePropertyName()).get();
            if (updateDayBudgetLastRunDate != null) {
                logger.info("Last run date is {}", updateDayBudgetLastRunDate);
                return updateDayBudgetLastRunDate;
            } else {
                logger.debug("Not found in db property with name {}", getLastRunDatePropertyName().getName());
                return null;
            }
        } catch (IllegalArgumentException e) {
            logger.error("Invalid property dateTime format", e);
            return null;
        }
    }

    PpcPropertyName<LocalDate> getLastRunDatePropertyName() {
        return new PpcPropertyName<>(String.format(LAST_RUN_DATE_PROPERTY, getShard()), PpcPropertyType.LOCAL_DATE);
    }
}
