package ru.yandex.direct.jobs.autooverdraft;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import org.apache.commons.validator.routines.EmailValidator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.autooverdraftmail.AutoOverdraftMailNotificationType;
import ru.yandex.direct.core.entity.client.model.ClientFlags;
import ru.yandex.direct.core.entity.client.repository.ClientOptionsRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.sender.YandexSenderClient;
import ru.yandex.direct.sender.YandexSenderException;
import ru.yandex.direct.sender.YandexSenderTemplateParams;
import ru.yandex.direct.utils.FunctionalUtils;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;

import static ru.yandex.direct.common.db.PpcPropertyNames.CLIENTS_CAN_ENABLE_AUTO_OVERDRAFT_LAST_UPLOAD_TIME;
import static ru.yandex.direct.jobs.configuration.JobsEssentialConfiguration.DEFAULT_YT_CLUSTER;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;
import static ru.yandex.direct.ytwrapper.YtPathUtil.generatePath;

@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 2, minutes = 5),
        notifications = @OnChangeNotification(method = NotificationMethod.TELEGRAM,
                recipient = NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        ),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_1, GROUP_INTERNAL_SYSTEMS}
)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 2),
        needCheck = NonProductionEnvironment.class,
        tags = {DIRECT_PRIORITY_1, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION}
)
@Hourglass(periodInSeconds = 3600, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault

public class AutoOverdraftEmailSenderJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(AutoOverdraftEmailSenderJob.class);

    private static void logEvent(AutoOverdraftMailEvent event) {
        logger.info(JsonUtils.toJson(event));
    }

    private static final int YT_READ_CHUNK_SIZE = 10_000;
    private static final int EMAILS_SEND_CHUNK_SIZE = 1_000;
    private static final String CLIENT_ID = "ClientID";

    // Выгрузка формируется джобой AutoOverdraftClientsForTeaserJob
    private static final String PATH_PREFIX = "//home/direct";
    private static final String EXPORT_PATH = "/export/autooverdraft_clients_for_teaser";
    private static final YtTable TABLE = new YtTable(generatePath(PATH_PREFIX, EXPORT_PATH));

    private final ShardHelper shardHelper;
    private final ClientOptionsRepository clientOptionsRepository;
    private final YandexSenderClient senderClient;
    private final RbacService rbacService;
    private final UserService userService;
    private final AutoOverdraftMailTemplateResolver autoOverdraftMailTemplateResolver;
    private final YtProvider ytProvider;
    private final YtCluster ytCluster;
    private final PpcProperty<String> lastUploadTimeProp;

    private static final EmailValidator emailValidator = EmailValidator.getInstance();
    private static final AutoOverdraftMailNotificationType autoOverdraftIsAvailable =
            AutoOverdraftMailNotificationType.AUTO_OVERDRAFT_IS_AVAILABLE;

    @Autowired
    public AutoOverdraftEmailSenderJob(ShardHelper shardHelper, YandexSenderClient senderClient,
                                       ClientOptionsRepository clientOptionsRepository, RbacService rbacService,
                                       UserService userService,
                                       AutoOverdraftMailTemplateResolver autoOverdraftMailTemplateResolver,
                                       YtProvider ytProvider, @Qualifier(DEFAULT_YT_CLUSTER) YtCluster ytCluster,
                                       PpcPropertiesSupport ppcPropertiesSupport) {
        this.shardHelper = shardHelper;
        this.senderClient = senderClient;
        this.clientOptionsRepository = clientOptionsRepository;
        this.rbacService = rbacService;
        this.userService = userService;
        this.autoOverdraftMailTemplateResolver = autoOverdraftMailTemplateResolver;
        this.ytProvider = ytProvider;
        this.ytCluster = ytCluster;
        this.lastUploadTimeProp = ppcPropertiesSupport.get(CLIENTS_CAN_ENABLE_AUTO_OVERDRAFT_LAST_UPLOAD_TIME);
    }

    @Override
    public void execute() {
        YtOperator ytOperator = ytProvider.getOperator(ytCluster);

        if (!ytOperator.exists(TABLE)) {
            logger.error("Table for import {} doesn't exists", TABLE);
            setJugglerStatus(JugglerStatus.CRIT, "Table doesn't exists");
            return;
        }

        String uploadTime = ytOperator.readTableUploadTime(TABLE);
        if (isDataProcessed(uploadTime)) {
            logger.info("Table data already processed");
            return;
        }

        logger.debug("import data from table uploaded at {}", uploadTime);
        long count = ytOperator.readTableSnapshot(TABLE, new ClientsCanEnableAutoOverdraftTableRow(), this::ytRowMapper,
                this::sendEmails, YT_READ_CHUNK_SIZE);
        lastUploadTimeProp.set(uploadTime);
        logger.info("Successfully processed {} rows from table", count);
    }

    boolean isDataProcessed(String uploadTime) {
        String lastUploadTime = lastUploadTimeProp.get();
        return lastUploadTime != null && lastUploadTime.compareTo(uploadTime) >= 0;
    }

    private void sendEmails(List<ClientId> clients) {
        shardHelper.groupByShard(clients, ShardKey.CLIENT_ID, ClientId::asLong)
                .chunkedBy(EMAILS_SEND_CHUNK_SIZE)
                .forEach(this::sendEmailsChunkInShard);
    }

    private void sendEmailsChunkInShard(int shard, List<ClientId> clients) {
        if (shard < 1) {
            logger.warn("Skip saving: unknown shard {} for data {}", shard, clients);
            setJugglerStatus(JugglerStatus.WARN, "Got data with unknown shard, check log for details");
            return;
        }
        processClientsInShard(shard, clients);
    }

    private ClientId ytRowMapper(ClientsCanEnableAutoOverdraftTableRow row) {
        return ClientId.fromLong(row.getClientId());
    }


    private void processClientsInShard(int shard, List<ClientId> clients) {
        Set<ClientId> clientsNotNotified = clientOptionsRepository.getClientsAutoOverdraftNotNotified(shard, clients);
        // Письма отправляются только главным представителям
        Map<ClientId, Long> client2chiefUid = rbacService.getChiefsByClientIds(clientsNotNotified);
        Collection<User> users = userService.massGetUser(client2chiefUid.values());
        Map<ClientId, User> indexedUsers = FunctionalUtils.listToMap(users, User::getClientId);

        for (ClientId clientId : clientsNotNotified) {
            User user = indexedUsers.get(clientId);

            String email = user.getRecommendationsEmail() != null ? user.getRecommendationsEmail()
                    : user.getEmail();
            String campSlug =
                    autoOverdraftMailTemplateResolver.resolveTemplateId(autoOverdraftIsAvailable);
            if (!emailValidator.isValid(email)) {
                logEvent(AutoOverdraftMailEvent.autoOverdraftBadEmail(clientId, user, email));
                continue;
            }
            YandexSenderTemplateParams templateParams = new YandexSenderTemplateParams.Builder()
                    .withCampaignSlug(campSlug)
                    .withToEmail(email)
                    .withAsync(Boolean.TRUE)
                    .withArgs(ImmutableMap.of(CLIENT_ID, clientId.toString()))
                    .build();
            try {
                if (senderClient.sendTemplate(templateParams, YandexSenderClient::isInvalidToEmail)) {
                    clientOptionsRepository.setClientFlag(shard, clientId, ClientFlags.AUTO_OVERDRAFT_NOTIFIED);
                    logEvent(AutoOverdraftMailEvent
                            .mailSent(autoOverdraftIsAvailable,
                                    clientId,
                                    user, campSlug));
                } else {
                    logEvent(AutoOverdraftMailEvent.autoOverdraftBadEmail(clientId, user, email));
                }
            } catch (YandexSenderException e) {
                logEvent(AutoOverdraftMailEvent.senderError(autoOverdraftIsAvailable,
                        clientId,
                        user, campSlug, e.getMessage()));
            }
        }
    }
}

